from datetime import datetime
from enum import Enum
import logging
from typing import Dict, List, Tuple, Set

from staff.lib import waffle
import yenv

from staff.groups.models import Group, GROUP_TYPE_CHOICES
from staff.groups.service.datasource import get_service_members
from staff.lib.db import atomic
from staff.person.models import Staff

from staff.departments.models import (
    Department,
    HRProduct,
    DepartmentStaff,
    ValuestreamRoles,
    InstanceClass,
    DEPARTMENT_CATEGORY,
)
from staff.departments.controllers.value_streams_mock import (
    get_services_heads as get_services_heads_mock,
    remove_value_streams,
    remove_value_streams_roles,
    create_value_streams_services,
)


logger = logging.getLogger('staff.departments.value_streams_rollup')


class ValueStreamRoleId(Enum):
    HEAD = 1
    HRBP = 161

    if yenv.type == 'production':
        MANAGER = 4062
    else:
        MANAGER = 1158


def _get_services_person_by_role(service_ids: List[int], role_id: ValueStreamRoleId) -> List[Tuple[str, int]]:
    """Возвращает туплы (person_login, service_id) по руководителям запрашиваемых сервисов"""
    product_heads = get_service_members(
        lookup={
            'service__in': ','.join(str(s_id) for s_id in service_ids),
            'role': role_id.value,
            'is_exportable': True,
            'fields': 'person.login,service.id',
            'page_size': 100,
        },
        timeout=(2, 5, 10),
    )
    for record in product_heads:
        yield record['person']['login'], record['service']['id']


class ValueStreamsRollupController:
    services_qs = Group.objects.filter(intranet_status=1, type=GROUP_TYPE_CHOICES.SERVICE)

    @atomic
    def rollup(self):
        if self._should_rollup_from_mock():
            remove_value_streams_roles()
            remove_value_streams()
            create_value_streams_services()

        self.rollup_value_stream_tree()
        self.rollup_value_stream_roles(ValueStreamRoleId.HEAD, ValuestreamRoles.HEAD)
        self.rollup_value_stream_roles(ValueStreamRoleId.MANAGER, ValuestreamRoles.MANAGER)
        self.rollup_value_stream_roles(ValueStreamRoleId.HRBP, ValuestreamRoles.HRBP)

    @staticmethod
    def _should_rollup_from_mock() -> bool:
        if yenv.type == 'production':
            return False

        if waffle.switch_is_active('enable_vs_rollup_from_groups'):
            return False

        return True

    def _get_services_roles(
        self,
        service_ids: List[int],
        role_id: ValueStreamRoleId,
    ) -> List[Tuple[str, int]]:
        if self._should_rollup_from_mock():
            return get_services_heads_mock(role_id)

        return _get_services_person_by_role(service_ids, role_id)

    def rollup_value_stream_tree(self):
        """
        Приводим дерево наших vs в соответствие с HR Product из OEBS
        """
        service_ids = self._get_all_service_ids()
        vs_groups_by_url = {
            g.url: g
            for g in self.services_qs.filter(service_id__in=service_ids)
        }

        vs_departments = Department.valuestreams.all()  # вместе с intranet_status=0
        vs_departments_by_url = {vs.url: vs for vs in vs_departments}

        # 1. Ищем соответствие сервисов с vs через поле url
        to_update = set(vs_groups_by_url) & set(vs_departments_by_url)
        to_create = set(vs_groups_by_url) - set(vs_departments_by_url)
        to_delete = set(
            vs_departments
            .filter(intranet_status=1)
            .exclude(url__in=vs_groups_by_url)
            .values_list('url', flat=True)
        )

        # 2. Досоздаём vs по сервисам, для которых соответствие не найдено
        created: Dict[str, Department] = {}
        for url in to_create:
            vs_group = vs_groups_by_url[url]
            parent_vs: Department = self._find_parent_vs(vs_group.parent_service_id)
            new_vs: Department = self._create_vs(vs_group, parent_vs=parent_vs)
            created[new_vs.url] = new_vs

            if parent_vs is None:
                vs_departments_by_url[new_vs.url] = new_vs
                to_update.add(new_vs.url)  # для случая, если parent_vs создастся в следующей итерации.

        # 3. Удаляем vs, для которых соответствия не найдено.
        for url in to_delete:
            vs_departments_by_url[url].intranet_status = 0
            vs_departments_by_url[url].save()

        # 4. Апдейтим vs, для которых найдено соответствие
        to_save: Dict[str, Department] = {}
        for url in to_update:
            if self.need_update(vs_groups_by_url[url], vs_departments_by_url[url]):
                vs_group = vs_groups_by_url[url]
                vs = vs_departments_by_url[url]

                parent_vs = self._find_parent_vs(vs_group.parent_service_id)
                self._update_vs(vs, vs_group, parent_vs)
                to_save[vs.url] = vs

        for url, vs in to_save.items():
            vs.save()

        # 5. Апдейтим HRProduct, для которых найдено соответствие
        for url, vs in created.items():
            vs_group: Group = vs_groups_by_url[url]
            product = HRProduct.objects.filter(service_id=vs_group.service_id).first()
            if product:
                product.value_stream = vs
                product.save()

        logger.info('Created %s Value Streams: %s', len(created), list(created.keys()))
        logger.info('Updated %s Value Streams: %s', len(to_save), list(to_save.keys()))
        logger.info('Deleted %s Value Streams: %s', len(to_delete), list(to_delete))

    def _get_all_service_ids(self) -> Set[int]:
        all_service_ids = set(HRProduct.objects.active().values_list('service_id', flat=True))
        missing_ids = all_service_ids

        while missing_ids:
            parent_service_ids = set(
                self.services_qs
                .filter(service_id__in=missing_ids)
                .exclude(parent_service_id__isnull=True)
                .values_list('parent_service_id', flat=True)
            )
            missing_ids = parent_service_ids - all_service_ids
            all_service_ids = all_service_ids.union(missing_ids)

        return all_service_ids

    @atomic
    def rollup_value_stream_roles(self, abc_role_id: ValueStreamRoleId, role_id: ValuestreamRoles):
        value_streams_qs = Department.valuestreams.filter(intranet_status=1)
        service_id_to_url = dict(
            self.services_qs
            .filter(url__in=value_streams_qs.values_list('url', flat=True))
            .values_list('service_id', 'url')
        )

        if not service_id_to_url:
            logger.warning('No value streams. Skip rolling up heads.')
            return

        service_url_to_vs_id = dict(
            Department.valuestreams
            .filter(intranet_status=1)
            .values_list('url', 'id')
        )
        current_department_staffs = {
            (ds.staff_id, ds.department_id): ds
            for ds in DepartmentStaff.objects.filter(role_id=role_id)
        }
        login_to_id_map = dict(Staff.objects.all().values_list('login', 'id'))

        created_count = 0
        for person_login, service_id in self._get_services_roles(list(service_id_to_url), abc_role_id):
            person_id = login_to_id_map.get(person_login, None)
            if person_id is None:
                logger.error('Login %s from abc not found in staff', person_login)
                continue
            service_url = service_id_to_url[service_id]
            vs_id = service_url_to_vs_id[service_url]
            ds = current_department_staffs.pop((person_id, vs_id), None)
            if not ds:
                DepartmentStaff.objects.create(
                    staff_id=person_id,
                    department_id=vs_id,
                    role_id=role_id,
                )
                logger.info('Created new Value Stream CHIEF: person %s, vs %s (%s)', person_id, service_url, vs_id)
                created_count += 1

        deleted_count = 0
        for (person_id, dep_id), ds in current_department_staffs.items():
            logger.info('Deleting Value Stream CHIEF: person %s, vs %s', person_id, dep_id)
            ds.delete()
            deleted_count += 1

        logger.info(
            'rollup_value_stream_roles finished. Added %s heads, deleted %s heads',
            created_count,
            deleted_count,
        )

    def _find_parent_vs(self, parent_service_id: int or None) -> Department or None:
        """По parent_service_id находим сервисную группу родительского VS и по ней находим сам VS"""
        if parent_service_id is None:
            return None
        return (
            Department.valuestreams
            .filter(url__in=self.services_qs.filter(service_id=parent_service_id).values_list('url', flat=True))
            .first()
        )

    def need_update(self, vs_group: Group, vs_department: Department) -> bool:
        differs = (
            vs_department.intranet_status != vs_group.intranet_status
            or vs_department.code != vs_group.code[:20]
            or vs_department.native_lang != vs_group.native_lang
            or vs_department.name != vs_group.name
            or vs_department.name_en != (vs_group.name_en or vs_group.name)
            or vs_department.description != vs_group.description
            or vs_department.description_en != vs_group.description
            or vs_department.parent != self._find_parent_vs(vs_group.parent_service_id)
        )
        return differs

    @staticmethod
    def _update_vs(vs_department: Department, vs_group: Group, parent_vs: Department or None) -> None:
        vs_department.intranet_status = vs_group.intranet_status
        vs_department.code = vs_group.code[:20]
        vs_department.native_lang = vs_group.native_lang
        vs_department.name = vs_group.name
        vs_department.name_en = vs_group.name_en or vs_group.name
        vs_department.description = vs_group.description
        vs_department.description_en = vs_group.description
        vs_department.parent = parent_vs

    @staticmethod
    def _create_vs(service: Group, parent_vs: Department = None) -> Department:
        now = datetime.now()

        value_stream = Department(
            instance_class=InstanceClass.VALUESTREAM.value,
            from_staff_id=0,
            code=service.code[:20],
            url=service.url,
            parent=parent_vs,
            native_lang=service.native_lang,
            name=service.name,
            name_en=service.name_en or service.name,
            description=service.description,
            description_en=service.description,
            kind_id=10,
            category=DEPARTMENT_CATEGORY.NONTECHNICAL,
            created_at=now,
            modified_at=now,
        )

        value_stream.save()
        return value_stream
