import calendar
import pymongo
from pymongo.write_concern import WriteConcern

from datetime import datetime, date

from django.conf import settings

from bson.objectid import ObjectId, InvalidId

from staff.lib.utils.date import parse_datetime
from staff.lib.mongodb import mongo, CollectionObjectGenerator


MONGO_COLLECTION_NAME = 'department_edit_proposals'


def _force_mongo_types(dict_object):
    def force_int(token):
        if isinstance(token, list):
            return [force_int(t) for t in token]
        if isinstance(token, dict):
            for key, value in token.items():
                token[key] = force_int(value)
        return token

    def force_datetime(token):
        if isinstance(token, date):
            return datetime(year=token.year, month=token.month, day=token.day)
        if isinstance(token, list):
            return [force_datetime(t) for t in token]
        if isinstance(token, dict):
            for key, value in token.items():
                token[key] = force_datetime(value)
        return token

    return force_int(force_datetime(dict_object))


def to_mongo_id(str_id: str) -> ObjectId:
    return ObjectId(str_id)


def from_mongo_id(obj_id: ObjectId) -> str:
    return str(obj_id)


def utc_to_local(utc_dt):
    """
    Приводит utc время к локальному для показа в интерфейсе
    utc_dt datetime или str в формате 'YYYY-MM-DDTHH:MM:SS.mmm'
    """
    if not utc_dt:
        return utc_dt
    string_format = isinstance(utc_dt, str)
    if string_format:
        utc_dt = parse_datetime(utc_dt)
    timestamp = calendar.timegm(utc_dt.timetuple())
    local_dt = datetime.fromtimestamp(timestamp)
    result = local_dt.replace(microsecond=utc_dt.microsecond)
    return result.strftime('%Y-%m-%dT%H:%M:%S.%f')[:-3] if string_format else result


def get_mongo_object(_id, fields=None):
    collection = mongo.db.get_collection(MONGO_COLLECTION_NAME)
    try:
        if not fields:
            return collection.find_one({'_id': to_mongo_id(_id)})
        else:
            fields = {f: 1 for f in fields}
            fields['_id'] = 0
            return collection.find_one({'_id': to_mongo_id(_id)}, fields)
    except InvalidId:
        return None


def get_mongo_objects(spec, sort=None, skip=0, limit=50):
    collection = mongo.db.get_collection(MONGO_COLLECTION_NAME)
    obj_generator = CollectionObjectGenerator(collection, spec)

    if sort and isinstance(sort, (list, dict, tuple)):
        sort = [
            (f.strip('-'), pymongo.DESCENDING if f.startswith('-') else pymongo.ASCENDING)
            for f in sort
        ]

    if sort:
        obj_generator = obj_generator.sort(sort)

    return obj_generator.iterator(skip=skip, limit=limit)


def save_mongo_object(obj_dict):
    collection = mongo.db.get_collection(
        MONGO_COLLECTION_NAME,
        write_concern=WriteConcern(settings.MONGO_WRITE_CONCERN_W),
    )
    return from_mongo_id(
        collection.insert_one(_force_mongo_types(obj_dict)).inserted_id
    )


def update_mongo_object(_id, obj_dict):
    collection = mongo.db.get_collection(
        MONGO_COLLECTION_NAME,
        write_concern=WriteConcern(settings.MONGO_WRITE_CONCERN_W),
    )
    obj_id = obj_dict.get('_id')
    if obj_id is not None and not isinstance(obj_id, ObjectId):
        obj_dict['_id'] = to_mongo_id(obj_id)

    collection.replace_one(
        {'_id': to_mongo_id(_id)}, _force_mongo_types(obj_dict),
        upsert=True
    )
    return _id


def update_mongo_field(_id, field_to_set, value):
    collection = mongo.db.get_collection(
        MONGO_COLLECTION_NAME,
        write_concern=WriteConcern(settings.MONGO_WRITE_CONCERN_W),
    )
    collection.update(
        {'_id': to_mongo_id(_id)},
        {'$set': {field_to_set: value}},
    )


def get_records_count(spec):
    collection = mongo.db.get_collection(MONGO_COLLECTION_NAME)
    obj_generator = CollectionObjectGenerator(collection, spec or {})
    return len(obj_generator)


def delete_mongo_object(object_id):
    collection = mongo.db.get_collection(MONGO_COLLECTION_NAME)
    if object_id is None:
        raise RuntimeError('Attempting to remove entire collection. Aborted.')
    return collection.delete_one({'_id': to_mongo_id(object_id)})
