import logging

from django.views.decorators.csrf import ensure_csrf_cookie
from django.views.decorators.http import require_http_methods

from staff.departments.models import Department
from staff.lib.decorators import responding_json, available_by_tvm
from staff.departments.models import ProposalMetadata
from staff.departments.edit.proposal_mongo import to_mongo_id
from staff.departments.controllers.proposal import ProposalCtl, proposal_oebs_format

logger = logging.getLogger('departments.edit_request_views')


@ensure_csrf_cookie
@available_by_tvm(['oebs-budgets'])
@require_http_methods(['GET'])
@responding_json
def proposal_data(request, proposal_id):
    """Информация о заявке для OEBS"""
    proposal_exists = ProposalMetadata.objects.filter(proposal_id=proposal_id).exclude(applied_at=None).exists()
    if not proposal_exists:
        return {'error': 'proposal has not been applied yet'}, 400

    proposal = ProposalCtl(proposal_id)
    proposal_data = proposal_oebs_format(proposal.proposal_object)
    return proposal_data


@require_http_methods(['GET'])
@responding_json
def moves_history(request, department_url):
    date_from = request.GET.get('from')
    date_to = request.GET.get('to')
    if date_from is None or date_to is None:
        return {
            'error': 'Both "from" and "to" parameters are required'
        }, 400

    try:
        department_id = (
            Department.objects
            .values_list('id', flat=True)
            .get(url=department_url)
        )
    except Department.DoesNotExist:
        return {
           'error': 'Department with url {} does not exist'.format(department_url)
        }, 404

    applied_proposals = dict(
        ProposalMetadata.objects
        .exclude(applied_at=None)
        .filter(
            applied_at__gte=date_from,
            applied_at__lte=date_to,
        )
        .values_list('proposal_id', 'applied_at')
    )

    proposals = ProposalCtl.filter(
        spec={
            '_id': {'$in': [to_mongo_id(_id) for _id in applied_proposals.keys()]},
            'actions.id': department_id,
        }
    )

    return {
        'result': [
            {
                'proposal_id': p.proposal_id,
                'applied_at': applied_proposals[p.proposal_id],
                'changing_duties': (
                    p.get_action(department_id)
                    .get('hierarchy', {})
                    .get('changing_duties')
                ),
            }
            for p in proposals
        ]
    }
