from contextlib import contextmanager

from django.contrib.auth.models import Permission
from django.contrib.postgres.fields import ArrayField
from django.db import models
from django.db.models.signals import pre_save
from django.utils.translation import ugettext_lazy as _

from staff.emission.logbroker.models import LogbrokerLog  # noqa ignore=F401
from staff.lib.models.base import (
    IntranetModel,
    IntranetMpttModel,
    IntranetMpttManager,
    SortedModel,
    TimeStampedModel,
    AtomicSaveModel,
)
from staff.lib.pg_lock import take_pg_lock
from staff.lib.utils.ordered_choices import OrderedChoices
from staff.lib.utils.enum_choices import EnumChoices


class DepartmentRoles(str, EnumChoices):
    CHIEF = 'C'
    DEPUTY = 'D'
    HR_PARTNER = 'H'
    BUDGET_HOLDER = 'B'
    GENERAL_DIRECTOR = 'G'
    HR_ANALYST = 'A'
    MARKET_ANALYST = 'MA'
    HR_ANALYST_TEMP = 'AT'
    CURATOR_BU = 'CURATOR_BU'
    CURATOR_EXPERIMENT = 'CURATOR_EXPERIMENT'
    FINCAB_VIEWER = 'FINCAB_VIEWER'


class ValuestreamRoles(str, EnumChoices):
    HEAD = 'VALUESTREAM_HEAD'
    MANAGER = 'VALUESTREAM_MANAGER'
    HRBP = 'VALUESTREAM_HRBP'


DEPARTMENT_CATEGORY = OrderedChoices(
    ('NONTECHNICAL', 'nontechnical', _('nontechnical category')),
    ('TECHNICAL', 'technical', _('technical category')),
)


class DepartmentTags(str, EnumChoices):
    business_unit = 'business_unit'
    experiment = 'experiment'


class InstanceClass(str, EnumChoices):
    DEPARTMENT = 'department'
    VALUESTREAM = 'valuestream'
    GEOGRAPHY = 'geography'


class DepartmentsManager(IntranetMpttManager):
    def get_queryset(self):
        return super().get_queryset().filter(instance_class=InstanceClass.DEPARTMENT.value)


class ValueStreamsManager(IntranetMpttManager):
    def get_queryset(self):
        return super().get_queryset().filter(instance_class=InstanceClass.VALUESTREAM.value)


class GeographyManager(IntranetMpttManager):
    def get_queryset(self):
        return super().get_queryset().filter(instance_class=InstanceClass.GEOGRAPHY.value)


class Department(IntranetMpttModel, SortedModel):

    class Meta(IntranetMpttModel.Meta):
        db_table = 'intranet_department'
        permissions = (
            ('can_view_departments', 'Can view departments'),
            ('can_view_profiles', 'Can view profiles'),
            # расширяет can_view_departments и can_view_profiles на все блоки (а не только с Block.external_other)
            ('can_view_all_blocks', 'Can view all blocks on permitted profiles'),
            ('can_view_gaps', 'Can view gaps'),
            ('can_view_finance', 'Can view finance cab'),
        )

    class MPTTMeta:
        order_insertion_by = ['position']

    from_staff_id = models.PositiveIntegerField(db_index=True)
    parent = models.ForeignKey('self', null=True, blank=True, related_name='children')
    kind = models.ForeignKey('DepartmentKind', null=True)

    name = models.CharField(max_length=255)
    name_en = models.CharField(max_length=255, default='')
    short_name = models.CharField(max_length=255, default='')
    short_name_en = models.CharField(max_length=255, default='')

    version = models.IntegerField(default=1)
    code = models.CharField(max_length=20, default='')
    url = models.CharField(max_length=255, default='', unique=True)
    bg_color = models.CharField(max_length=20, default='')
    fg_color = models.CharField(max_length=20, default='')

    maillists = models.CharField(max_length=4096, default='')
    clubs = models.CharField(max_length=4096, default='')
    wiki_page = models.CharField(max_length=2048, default='')
    jira_project = models.CharField(max_length=255, default='')
    description = models.CharField(max_length=4096, default='')
    description_en = models.CharField(max_length=4096, default='')

    # Потомки простите, я был слаб и не смог противостоять натиску требований.
    # Надеюсь в ваше время вы можете делать стройные и гибкие системы без костылей.
    is_expanded = models.BooleanField(default=False, help_text='Разворачивается при открытии интерфейса структуры')
    is_hidden = models.BooleanField(default=False, help_text='Ветка скрывается в интерфейсе структуры')
    is_cut_out = models.BooleanField(
        default=False,
        help_text='Вырезано из иерархии в интерфейсе, а дети подятнуты к родителю выше',
    )
    tags = ArrayField(
        base_field=models.CharField(max_length=64, choices=DepartmentTags.choices()),
        blank=True,
        null=True,
        help_text='Теги, которые выводятся в интерфейсе структуру, например `Бизнесс юнит`',
    )
    section_group = models.ForeignKey(
        'departments.DepartmentInterfaceSection',
        null=True,
        blank=True,
        related_name='+',
        on_delete=models.SET_NULL,
        help_text='Подразделение объединено в безымянную группу',
    )
    section_caption = models.ForeignKey(
        'departments.DepartmentInterfaceSection',
        null=True,
        blank=True,
        related_name='+',
        on_delete=models.SET_NULL,
        help_text='Подразделение объединено под заголовок',
    )
    # ---

    category = models.CharField(
        max_length=30,
        choices=DEPARTMENT_CATEGORY.choices(),
        default=DEPARTMENT_CATEGORY.NONTECHNICAL,
    )
    instance_class = models.CharField(
        max_length=16,
        choices=InstanceClass.choices(),
        default=InstanceClass.DEPARTMENT.value,
    )

    oebs_creation_date = models.DateField(null=True, blank=True)

    objects = DepartmentsManager()
    valuestreams = ValueStreamsManager()
    geography = GeographyManager()
    all_types = IntranetMpttManager()

    def __str__(self):
        return self.name

    @property
    def chief(self):
        try:
            return self.departmentstaff_set.select_related('staff').filter(role_id=DepartmentRoles.CHIEF.value)[0].staff
        except IndexError:
            return None

    @property
    def deputy(self):
        try:
            return (
                self.departmentstaff_set
                .select_related('staff')
                .filter(role_id=DepartmentRoles.DEPUTY.value)[0].staff
            )
        except IndexError:
            return None

    @classmethod
    def get_tree_ids(cls, urls: list) -> dict:
        return dict(
            cls.objects
            .filter(url__in=urls)
            .values_list('url', 'tree_id')
        )

    def save(self, *args, **kwargs):
        five_minutes = 60 * 5 * 1000
        take_pg_lock('departments_lock', five_minutes, five_minutes)
        with self._set_manager():
            super().save(*args, **kwargs)

    @property
    def instance_class_manager(self):
        managers = {
            InstanceClass.DEPARTMENT.value: self.__class__.objects,
            InstanceClass.VALUESTREAM.value: self.__class__.valuestreams,
            InstanceClass.GEOGRAPHY.value: self.__class__.geography,
        }
        return managers.get(self.instance_class, self.__class__.objects)

    @contextmanager
    def _set_manager(self):
        all_types_manager = self.__class__.all_types

        old_manager = self.__class__._default_manager
        old_tree_manager = self.__class__._tree_manager

        self.__class__._default_manager = all_types_manager
        self.__class__._tree_manager = all_types_manager
        try:
            yield
        finally:
            self.__class__._default_manager = old_manager
            self.__class__._tree_manager = old_tree_manager


class DepartmentKind(IntranetModel):
    class Meta(IntranetModel.Meta):
        db_table = 'intranet_department_kind'

    name = models.CharField(max_length=64)
    name_en = models.CharField(max_length=64, blank=True)
    rank = models.IntegerField(default=0)
    slug = models.CharField(
        max_length=64,
        null=True,
        unique=True,
    )

    def __str__(self):
        return '%s %s' % (self.slug or '', self.rank)


class DepartmentStaff(AtomicSaveModel):
    department = models.ForeignKey('Department', null=True, on_delete=models.CASCADE)
    staff = models.ForeignKey('Staff', null=True)
    role = models.ForeignKey('departments.DepartmentRole', on_delete=models.CASCADE)

    def __str__(self):
        return '%s is %s in %s' % (self.staff.login, self.role, str(self.department))

    class Meta:
        db_table = 'intranet_departmentstaff'
        app_label = 'django_intranet_stuff'


class DepartmentRole(SortedModel):
    """Роль в подразделении. Замена для enum."""
    id = models.CharField(max_length=32, primary_key=True)
    slug = models.CharField(max_length=32, default='', db_index=True)
    name = models.CharField(max_length=64, default='')
    name_en = models.CharField(max_length=64, default='')
    show_in_structure = models.BooleanField(default=False)
    manage_by_idm = models.BooleanField(default=True)
    is_active = models.BooleanField(default=True)
    permissions = models.ManyToManyField(Permission, related_name='department_roles', blank=True)
    holders = models.ManyToManyField(
        'django_intranet_stuff.Staff',
        through=DepartmentStaff,
        related_name='department_roles',
    )

    def __str__(self):
        return self.name_en


class DepartmentResponsible(DepartmentStaff):
    """Заместитель, ответственный за подразделение"""
    responsible_for = models.ForeignKey('Department', related_name='responsibles')

    def __str__(self):
        return '%s is responsible for %s in %s' % (
            self.staff.login, str(self.responsible_for), str(self.department)
        )

    class Meta:
        db_table = 'intranet_departmentresponsible'
        app_label = 'django_intranet_stuff'


class DepartmentStaffCounter(models.Model):
    department = models.ForeignKey('Department', null=False)
    day = models.DateField(db_index=True)
    staff_counter = models.PositiveSmallIntegerField()

    def __str__(self):
        return '%s at %s: %d' % (self.department.name, self.day, self.staff_counter)

    class Meta:
        unique_together = (('department', 'day'),)
        db_table = 'intranet_departmentstaffcounter'
        app_label = 'django_intranet_stuff'


class DepartmentInterfaceSection(TimeStampedModel):
    name = models.CharField(max_length=255, default='')
    name_en = models.CharField(max_length=255, default='')
    description = models.CharField(max_length=4096)
    order = models.IntegerField(default=0)

    def __str__(self):
        return self.description


class DepartmentChain(AtomicSaveModel):
    """
        Немного денормализации для person.chiefs и person.hr_partners в staff-api.
        Не надо использовать эту табличку в стаффе.
    """
    department = models.OneToOneField(Department)
    chiefs = ArrayField(models.IntegerField())
    hr_partners = ArrayField(models.IntegerField())


def update_version(sender, instance, using, **kwargs):
    if sender == Department:
        instance.version += 1


pre_save.connect(update_version, weak=False)
