import logging

from django.conf import settings

from staff.celery_app import app
from staff.groups.models import Group, GROUP_TYPE_CHOICES
from staff.lib import requests
from staff.lib.db import atomic
from staff.lib.sync_tools.st_translation_sync import FieldSyncParams, StartrekFieldsSync
from staff.lib.tasks import LockedTask
from staff.lib.tvm2 import TVM_SERVICE_TICKET_HEADER, get_tvm_ticket_by_deploy

from staff.departments.controllers import proposal
from staff.departments.edit.proposal_mongo import to_mongo_id
from staff.departments.update_dep_chains import UpdateDepartmentChains  # noqa for celery task discovery
from staff.departments.models import Department, ProposalMetadata, Geography


logger = logging.getLogger('staff.departments.tasks')


@app.task(ignore_result=True)
class PushProposalsToOEBS(LockedTask):
    """Таска по очереди пушит _id всех выполненных но ещё не запушенных заявок в ОЕБС"""

    @staticmethod
    def push_proposal(proposal):
        oebs_response = requests.post(
            url=settings.OEBS_PUSH_PROPOSAL_URL,
            json={'eventID': proposal.proposal_id},
            headers={TVM_SERVICE_TICKET_HEADER: get_tvm_ticket_by_deploy('oebs-api')},
            log_message='Pushing proposal {proposal_id} to oebs'.format(proposal_id=proposal.proposal_id),
            timeout=(1, 3),
        )
        if oebs_response.status_code == 200:
            proposal.mark_pushed()
            return True
        else:
            logger.error(
                'Proposal %s push to oebs failed. Status code: %s; response: %s',
                proposal.proposal_id,
                oebs_response.status_code,
                oebs_response.text,
            )
            return False

    @staticmethod
    def get_proposals_to_push():
        from staff.departments.edit.constants import HR_CONTROLLED_DEPARTMENT_ROOTS
        finished_proposals_data = dict(
            ProposalMetadata.objects
            .filter(pushed_to_oebs=None)
            .exclude(applied_at=None)
            .values_list('proposal_id', 'applied_at')
        )
        proposals = proposal.ProposalCtl.filter(
            spec={
                '_id': {'$in': [to_mongo_id(_id) for _id in finished_proposals_data]},
                '$or': [
                    {'root_departments': root_dep}
                    for root_dep in dict(HR_CONTROLLED_DEPARTMENT_ROOTS)
                ]
            },
        )
        dates_with_proposals = sorted(
            (finished_proposals_data[p.proposal_id], p)
            for p in proposals
        )
        for apply_date, proposal_ in dates_with_proposals:
            yield proposal_

    def locked_run(self, *args, **kwargs):
        for proposal_to_push in self.get_proposals_to_push():
            if not self.push_proposal(proposal_to_push):
                return  # заканчиваем работу на первой незапушенной заявке


@app.task
class FixGroupsParents(LockedTask):
    def locked_run(self):
        check_and_fix_groups_parents()


@atomic
def check_and_fix_groups_parents():
    root_group_id = Group.objects.get(url='__departments__').id

    dep_parent_by_dep = {
        dep_id: dep_parent_id
        for dep_id, dep_parent_id in
        Department.objects.values_list('id', 'parent_id')
    }

    groups = (
        Group.objects
        .select_for_update()
        .filter(type=GROUP_TYPE_CHOICES.DEPARTMENT)
        .exclude(id=root_group_id)
        .values_list(
            'id',
            'parent_id',
            'department_id',
        )
        .order_by('tree_id', 'lft')
    )

    group_by_dep = {
        group_dep_id: group_id
        for group_id, group_parent_id, group_dep_id in groups
    }

    for group_id, group_parent_id, group_dep_id in groups:

        group_dep_parent_id = dep_parent_by_dep[group_dep_id]
        if group_dep_parent_id is None:
            correct_group_parent_id = root_group_id
        else:
            correct_group_parent_id = group_by_dep[group_dep_parent_id]

        if group_parent_id != correct_group_parent_id:
            try:
                fix_group_parent(group_id, correct_group_parent_id)
            except Exception:
                logger.exception(
                    'Fix group %s parent %s', group_id, correct_group_parent_id
                )
                raise
            else:
                logger.info(
                    'Group %s wrong parent = %s, fix on %s',
                    group_id, group_parent_id, correct_group_parent_id
                )


def fix_group_parent(group_id, group_parent_id):
    groups = Group.objects.in_bulk([group_id, group_parent_id])
    groups[group_id].parent = groups[group_parent_id]
    groups[group_id].save()
    children_groups = groups[group_id].get_descendants().order_by('level')
    for child in children_groups:
        child.save()


@app.task(ignore_result=True)
class SyncGeographyFieldTask(LockedTask):
    def locked_run(self, *args, **kwargs):
        params = FieldSyncParams(
            qs=Geography.objects.filter(intranet_status=1),
            key_prefix='oebs',
            key_field='department_instance.url',
            translation_ru_field='name',
            translation_en_field='name_en',
            startrek_field='geography2',
        )

        sync = StartrekFieldsSync(params)
        sync.sync()
