from functools import wraps

from django import template
from django.conf import settings
from django.core.urlresolvers import reverse

from staff.femida.constants import VACANCY_STATUS
from staff.oebs.constants import PERSON_POSITION_STATUS


register = template.Library()


def maybe_empty(func):
    empty_value = '%%—%%'

    @wraps(func)
    def decorated_func(param):
        if param is None:
            return empty_value
        return func(param)
    return decorated_func


def maybe_list(func):
    delimiter = ' '

    @wraps(func)
    def decorated_func(param):
        if param is not None and isinstance(param, list):
            return delimiter.join([func(p) for p in param])
        return func(param)
    return decorated_func


def wiki_formatted_url(url, title=None):
    return '(({url} {title}))'.format(
        url=url,
        title=title or url
    )


@register.filter
def femida_vacancy_url(vacancy_id):
    return 'https://{femida_host}/vacancies/{vacancy_id}/'.format(
        femida_host=settings.FEMIDA_HOST,
        vacancy_id=vacancy_id,
    )


@register.filter
def translate_headcount_status(status):
    return {
        PERSON_POSITION_STATUS.OCCUPIED: 'Работает',
        PERSON_POSITION_STATUS.OFFER: 'Оффер',
        PERSON_POSITION_STATUS.VACANCY_OPEN: 'Открытая вакансия',
        PERSON_POSITION_STATUS.VACANCY_PLAN: 'Незанятый хедкаунт',
        PERSON_POSITION_STATUS.RESERVE: 'Кредит',
        PERSON_POSITION_STATUS.MATERNITY: 'Декрет',
    }.get(status, '')


@register.filter
def translate_vacancy_status(status):
    return {
        VACANCY_STATUS.CLOSED: 'Закрыта',
        VACANCY_STATUS.DRAFT: 'Черновик',
        VACANCY_STATUS.ON_APPROVAL: 'На согласовании',
        VACANCY_STATUS.OFFER_ACCEPTED: 'Оффер принят',
        VACANCY_STATUS.SUSPENDED: 'Приостановлена',
        VACANCY_STATUS.IN_PROGRESS: 'В работе',
        VACANCY_STATUS.OFFER_PROCESSING: 'Оффер',
    }.get(status, '')


@register.filter
def translate_vacancy_simple_status(status):
    return {
        VACANCY_STATUS.CLOSED: 'Вакансия',
        VACANCY_STATUS.DRAFT: 'Вакансия',
        VACANCY_STATUS.ON_APPROVAL: 'Вакансия',
        VACANCY_STATUS.OFFER_ACCEPTED: 'Оффер',
        VACANCY_STATUS.SUSPENDED: 'Вакансия',
        VACANCY_STATUS.IN_PROGRESS: 'Вакансия',
        VACANCY_STATUS.OFFER_PROCESSING: 'Оффер',
    }.get(status, '')


@register.filter
def staff_login(login):
    if not login:
        return ''

    return f'https://{settings.STAFF_HOST}/{login}/'


@register.filter
def abc_service(service: str or int):
    if not service:
        return ''

    return f'https://{settings.ABC_FRONT_HOST}/services/{service}'


@register.filter
@maybe_list
@maybe_empty
def plain_text(text):
    return '%%{text}%%'.format(text=text)


@register.filter
@maybe_list
@maybe_empty
def person(staff_instance):
    return 'кто:{login}'.format(login=staff_instance.login)


@register.filter
@maybe_list
@maybe_empty
def department(department_instance):
    if isinstance(department_instance, str):
        return department_instance
    elif isinstance(department_instance, dict):
        dep_url = department_instance.get('url')
    else:
        dep_url = department_instance.url

    url = 'https://{host}{url}'.format(
        host=settings.STAFF_HOST,
        url=reverse(
            'departments-frontend:department',
            kwargs={'url': dep_url},
        ),
    )
    return url
