import statistics
from collections import defaultdict
from datetime import date
from typing import List, Dict, Set, Iterable

from django.conf import settings
from django.db.models import F
from django.db.models.query import ValuesListQuerySet, QuerySet

from staff.lib.calendar import get_holidays
from staff.map.models import StaffOfficeLog


class ExportAttendanceAccessException(Exception):

    def __init__(self, field, message):
        self.field = field
        self.message = message

    def __str__(self):
        return self.message


class ExportAttendance(object):

    date_from: date
    date_to: date

    def __init__(self, queryset: QuerySet, date_from: date, date_to: date):
        self.queryset = queryset
        self.date_from = date_from
        self.date_to = date_to

    def _get_staff_geo_ids(self) -> Dict[int, int]:
        queryset = (
            self.queryset
            .values_list('id', 'office__city__geo_id')
        )
        return dict(queryset)

    def _get_work_days(self, geo_ids: Set[int]) -> Dict[int, List[date]]:
        work_days = {}
        calendar_work_days = get_holidays(self.date_from, self.date_to, geo_ids, out_mode='weekdays')
        for geo_id in calendar_work_days:
            work_days[geo_id] = [day['date'] for day in calendar_work_days[geo_id]]
        return work_days

    def _get_office_log_qs(self, staff_ids: Iterable[int]) -> ValuesListQuerySet:
        return (
            StaffOfficeLog
            .objects
            .filter(
                staff__id__in=staff_ids,
                date__gte=self.date_from,
                date__lte=self.date_to,
                office_id=F('staff__office_id'),
            )
            .values_list('date', 'staff_id', 'office__city__geo_id')
        )

    def _calculate_attendance(
            self,
            staff_geo_ids: Dict[int, int],
            work_days: Dict[int, List[date]]
    ) -> Dict[str, float]:
        staff_ids = staff_geo_ids.keys()
        office_log_qs = self._get_office_log_qs(staff_ids)
        attendance_result = defaultdict(float)
        for office_log_date, staff_id, geo_id in office_log_qs:
            if office_log_date not in work_days[geo_id or settings.RUSSIA_GEO_ID]:
                continue
            attendance_result[staff_id] += 1 / len(work_days[geo_id])
        attendance_result = list(attendance_result.values())
        absent_staff_count = len(staff_ids) - len(attendance_result)
        attendance_result += [0] * absent_staff_count

        median = statistics.median(attendance_result)
        average = statistics.mean(attendance_result)

        return {
            'average': round(average, 2),
            'median': round(median, 2),
        }

    def export(self) -> Dict[str, float]:
        staff_geo_ids = self._get_staff_geo_ids()
        staff_ids = staff_geo_ids.keys()
        if len(staff_ids) < settings.EXPORT_ATTENDANCE_MIN_FILTER_PERSONS:
            raise ExportAttendanceAccessException(field='filter', message='too_few_persons')
        geo_ids = set(staff_geo_ids.values())
        work_days = self._get_work_days(geo_ids)
        if any(len(work_days[geo_id]) < settings.EXPORT_ATTENDANCE_MIN_PERIOD_DAYS for geo_id in work_days):
            raise ExportAttendanceAccessException(field='date_to', message='period_too_short')
        return self._calculate_attendance(
            staff_geo_ids=staff_geo_ids,
            work_days=work_days,
        )
