from collections import defaultdict
from itertools import chain

from django.db.models.query import QuerySet
from django.db import connections, router
from typing import Iterable, Dict

from staff.headcounts.permissions import Permissions
from staff.person_filter.filter_context import FilterContext
from staff.lib.utils.qs_values import extract_related
from staff.person.models import Staff

from staff.departments.models import Department, DepartmentStaff, DepartmentRoles
from staff.departments.tree_lib import AbstractEntityInfo
from staff.departments.tree.persons_list_filler import PersonsListFiller


PLURAL_KEYS = {
    'deputy': 'deputies',
    'hr_partner': 'hr_partners',
    'budget_holder': 'budget_holders',
}


class PersonsEntityInfo(AbstractEntityInfo):
    def __init__(self, filter_context: FilterContext, viewer_person: Staff = None):
        super().__init__(filter_context)

        self.permissions = Permissions(viewer_person) if viewer_person else None
        self._chiefs = None
        self._cached_chief_deps_ids = set()

    @staticmethod
    def _create_cursor():
        return connections[router.db_for_read(Department)].cursor()

    def _departments_available_for_viewer(self, ids):
        # type: (Iterable) -> set
        if not self.permissions:
            return set()

        return set(self.permissions.filter_by_observer(
            Department.objects.filter(id__in=ids).values_list('id', flat=True)
        ))

    def _get_person_sql(self):
        person_sql, params = (
            self.filter_context.get_person_qs()
            .values('id')
            .query.sql_with_params()
        )
        start_person_sql = 'SELECT `intranet_staff`.`id` FROM `intranet_staff`'
        return person_sql[len(start_person_sql):], params

    def _get_deep_persons_qty(self, all_ids) -> Dict[int, int]:
        sql = """
        SELECT dep.id, COUNT(1)
        FROM intranet_department AS dep
        INNER JOIN intranet_department AS p_dep ON (
            dep.id IN ({ids_places})
            AND dep.lft <= p_dep.lft
            AND dep.rght >= p_dep.rght
            AND dep.tree_id = p_dep.tree_id
        )
        INNER JOIN intranet_staff ON (
            intranet_staff.department_id = p_dep.id
        )
        {person_sql}
        GROUP BY dep.id
        """

        person_sql, params = self._get_person_sql()
        params = tuple(all_ids) + params
        sql = sql.format(
            person_sql=person_sql,
            ids_places=', '.join(['%s'] * len(all_ids))
        )

        cursor = self._create_cursor()
        cursor.execute(sql, params)
        return {d[0]: d[1] for d in cursor.fetchall()}

    def _get_vacancies_count(self, all_ids) -> Dict[int, int]:
        """Количество вакансий в сруктуре подразделении каждого подразделений"""
        sql = """
        SELECT dep.id, COUNT(departments_vacancy.id)
        FROM intranet_department AS dep
        INNER JOIN intranet_department AS v_dep ON (
            dep.id IN ({ids_places})
            AND dep.lft <= v_dep.lft
            AND dep.rght >= v_dep.rght
            AND dep.tree_id = v_dep.tree_id
        )
        INNER JOIN departments_vacancy ON (
            departments_vacancy.department_id = v_dep.id
        )
        WHERE
            departments_vacancy.status IN ('in_progress', 'offer_processing', 'suspended')
            AND departments_vacancy.is_published = true
        GROUP BY dep.id
        """

        params = tuple(all_ids)
        sql = sql.format(ids_places=', '.join(['%s'] * len(all_ids)))

        cursor = self._create_cursor()
        cursor.execute(sql, params)

        return {d[0]: d[1] for d in cursor.fetchall()}

    def fill_counters(self, all_ids, departments):
        deep_persons_qty = self._get_deep_persons_qty(all_ids)
        vacancies_count_map = self._get_vacancies_count(all_ids)
        departments_headcounts_available = self._departments_available_for_viewer(all_ids)

        for dep in departments:
            dep['persons_qty'] = deep_persons_qty.get(dep['id'], 0)
            dep['vacancies_count'] = vacancies_count_map.get(dep['id'], 0)
            dep['headcounts_available'] = dep['id'] in departments_headcounts_available

    def _get_chiefs(self, all_ids):
        """Руководители и другие роли"""
        all_ids = set(all_ids)
        if not all_ids - self._cached_chief_deps_ids:
            return self._chiefs
        self._cached_chief_deps_ids.update(all_ids)

        fields = ['staff__' + f for f in self.filter_context.short_person_fields]
        fields += ['department', 'role__slug', 'role__position', 'role__name', 'role__name_en']
        roles = (
            DepartmentStaff.objects
            .filter(
                department_id__in=self._cached_chief_deps_ids,
                staff__is_dismissed=False,
                role__show_in_structure=True,
            )
            .values(*fields)
            .order_by('role__position', 'staff__last_name')
        )
        roles = PersonsListFiller(self.filter_context).get_chiefs_as_list(roles)

        self._chiefs = defaultdict(list)
        for person_role in roles:
            person_role['person'] = extract_related(person_role, 'staff')
            person_role['role'] = extract_related(person_role, 'role')
            dep = person_role.pop('department')
            self._chiefs[dep].append(person_role)

        return self._chiefs

    def _set_crown_and_sort(self, persons, chiefs):
        # Удаляем из списка ролей присутствующих в списке людей
        # Добавляем в человека короны, если они есть
        # сортируем по коронам и фамилии список людей
        for person in persons:
            chief_info = chiefs.pop(person['id'], [])
            for role in chief_info:
                person.setdefault('all_crowns', []).append(role['role']['slug'])
                if 'crown' not in person:
                    person['crown'] = role['role']['slug']
                    person['role_position'] = role['role']['position']  # временная штука удалится при сортировке

        def sort_func(person):
            crown = person.get('crown')
            index = str(person.pop('role_position')) if crown else 'z'
            return index + person.get('last_name')

        return sorted(persons, key=sort_func)

    def fill_info(self, all_ids, departments, info_map):
        chiefs_map = self._get_chiefs(all_ids)

        for dep in departments:
            info = dep.setdefault('info', {})
            chiefs = chiefs_map.get(dep['id'], [])
            chiefs_by_person = defaultdict(list)

            persons = (
                info_map[dep['id']]
                if info_map is not None and dep['id'] in info_map
                else []
            )
            persons_ids = [p['id'] for p in persons]

            for role_owner in chiefs:
                role_owner['member_of_department'] = role_owner['person']['id'] in persons_ids
                info.setdefault('roles', []).append(role_owner)
                chiefs_by_person[role_owner['person']['id']].append(role_owner)

            if persons:
                info['persons'] = self._set_crown_and_sort(persons, chiefs_by_person)

            for role_owner in chain.from_iterable(chiefs_by_person.values()):
                role = role_owner['role']['slug']
                person = role_owner['person']

                if role in ('chief', 'general_director'):
                    info[role] = person
                else:
                    key = PLURAL_KEYS.get(role, role)
                    info.setdefault(key, []).append(person)

            if 'persons_qty' in dep:
                info['persons_qty'] = dep['persons_qty']
            if 'vacancies_count' in dep:
                info['vacancies_count'] = dep['vacancies_count']

        return departments

    def fill_dep_attrs(self, all_ids, departments):
        chiefs_map = self._get_chiefs(all_ids)

        for dep in departments:
            chiefs = chiefs_map.get(dep['id'], [])
            roles = {r['role']['slug'] for r in chiefs}
            dep['is_experiment'] = DepartmentRoles.CURATOR_EXPERIMENT.value.lower() in roles
            dep['is_bu'] = DepartmentRoles.CURATOR_BU.value.lower() in roles

        return departments

    def filter_result(self, departments):
        if not self.filter_context.filter_id:
            return departments

        return [d for d in departments if d['persons_qty']]

    def filter(self, all_ids, departments):
        if not self.filter_context.filter_id:
            return departments

        deep_persons_qty = self._get_deep_persons_qty(all_ids)
        return [d for d in departments if deep_persons_qty.get(d['id'])]

    def order_entities_by_fields(self):
        return []

    def fill_list(self, entities):
        return PersonsListFiller(self.filter_context).get_as_list(entities)

    def departments_query(self):
        return self.filter_context.get_base_dep_qs().order_by('tree_id', 'level', 'lft')

    def entities_quantity_by_department_query(self, entity_dep_field_name: str) -> QuerySet:
        field_name = entity_dep_field_name.replace('_id', '')
        return (
            self.filter_context
            .get_persons_qty_qs()
            .order_by(f'{field_name}__tree_id', f'{field_name}__level', f'{field_name}__lft')
        )

    def short_entities_query(self):
        return self.filter_context.get_person_qs()

    def full_entities_query(self):
        return self.filter_context.get_base_person_qs()

    def total_entities_aggregate_query(self, fields):
        return self.filter_context.get_person_qs(fields)

    def total_entities_count_query(self):
        return self.total_entities_aggregate_query(['id'])


class ProposalPersonsEntityInfo(PersonsEntityInfo):
    def fill_dep_attrs(self, all_ids, departments):
        return departments
