from itertools import chain

from django.db.models import Q

from staff.departments.models import Department

from staff.departments.tree_lib.tree_builder import TreeBuilder
from staff.departments.tree_lib.abstract_entity_info import AbstractEntityInfo


class TreeGetter(object):
    def __init__(self, info_provider: AbstractEntityInfo):
        self.info_provider = info_provider

    def get_default(self):
        return self.info_provider.departments_query().filter(parent_id=None)

    def get_by_url(self, url):
        try:
            dep = self.info_provider.departments_query().get(url=url)
        except Department.DoesNotExist:
            return []

        dep['is_selected'] = True
        result = self._with_ancestors([dep])
        return result

    def search(self, q):
        # Найти подразделения
        found_departments = (
            self.info_provider.departments_query()
            .filter(Q(url=q) | Q(name_en__icontains=q) | Q(name__icontains=q))
            .order_by('tree_id', 'lft')
        )

        found_departments = TreeBuilder(self.info_provider).filter(found_departments)

        extra = {'relevant_qty': len(found_departments)}

        if not found_departments or len(found_departments) > 50:
            return [], extra

        extra['finded_urls'] = [d['url'] for d in found_departments]

        for dep in found_departments:
            dep['is_relevant'] = True

        result = self._with_ancestors(found_departments)
        return result, extra

    def _with_ancestors(self, departments):
        parents_q = Q()
        found_ids = []
        for dep in departments:
            parents_q |= Q(
                tree_id=dep['tree_id'],
                lft__lt=dep['lft'],
                rght__gt=dep['rght']
            )
            found_ids.append(dep['id'])

        # Получить родителей
        parents_deps = (
            self.info_provider
            .departments_query()
            .filter(parents_q)
            .exclude(id__in=found_ids)
        )
        parents_ids = [p['id'] for p in parents_deps]

        # Получить соседей
        neighbors_deps = (
            self.info_provider
            .departments_query()
            .filter(Q(parent_id__in=parents_ids) | Q(parent_id=None))
            .exclude(id__in=chain(parents_ids, found_ids))
        )

        return list(chain(departments, parents_deps, neighbors_deps))
