from logging import getLogger
from collections import defaultdict

from staff.celery_app import app
from staff.lib.db import atomic
from staff.lib.tasks import LockedTask

from staff.departments.models import (
    Department,
    DepartmentStaff,
    DepartmentChain,
    DepartmentRoles,
)


logger = getLogger(__name__)


@app.task(ignore_result=True)
class UpdateDepartmentChains(LockedTask):
    def locked_run(self, *args, **kwargs):
        update_department_chains()


@atomic
def update_department_chains():
    created, updated, deleted = 0, 0, 0

    new_chains = get_department_chains()
    current_chains = DepartmentChain.objects.all()

    processed_departments = set()
    for chain in current_chains:
        processed_departments.add(chain.department_id)
        if chain.department_id in new_chains:
            should_save = False

            new_chiefs = new_chains[chain.department_id]['chiefs']
            if chain.chiefs != new_chiefs:
                chain.chiefs = new_chiefs
                should_save = True

            new_hr_partners = new_chains[chain.department_id]['hr_partners']
            if chain.hr_partners != new_hr_partners:
                chain.hr_partners = new_hr_partners
                should_save = True

            if should_save:
                chain.save()
                updated += 1
        else:
            chain.delete()
            deleted += 1

    for department_id, chains in new_chains.items():
        if department_id not in processed_departments:
            DepartmentChain.objects.create(
                department_id=department_id,
                chiefs=chains['chiefs'],
                hr_partners=chains['hr_partners'],
            )
            created += 1

    logger.info("Sync department chains: created %d, updated %d, deleted %d", created, updated, deleted)


def get_department_chains():
    dep_to_parent = dict(Department.objects.filter(intranet_status=1).values_list('id', 'parent_id'))
    depstaffs = (
        DepartmentStaff.objects
        .filter(role__in=[DepartmentRoles.CHIEF.value, DepartmentRoles.HR_PARTNER.value])
        .values('department_id', 'staff_id', 'role')
    )

    chiefs_by_dep = defaultdict(list)
    hr_partners_by_dep = defaultdict(list)

    for depstaff in depstaffs:
        department_id = depstaff['department_id']
        staff_id = depstaff['staff_id']
        if depstaff['role'] == DepartmentRoles.CHIEF.value:
            chiefs_by_dep[department_id].append(staff_id)
        elif depstaff['role'] == DepartmentRoles.HR_PARTNER.value:
            hr_partners_by_dep[department_id].append(staff_id)

    visited_departments = set()

    def process_department(department_id):
        visited_departments.add(department_id)

        parent_id = dep_to_parent[department_id]
        if not parent_id:
            return

        if parent_id not in visited_departments:
            process_department(parent_id)

        # тут могут быть сложные цепочки, когда один и тот же руководитель
        # встречается несколько раз, но в текущем стафф-апи люди так
        # их и забирают, поэтому пока оставляем
        chiefs_by_dep[department_id].extend(chiefs_by_dep[parent_id])

        # hr-партнеры могут повторяться, в стафф-апи их не повторяем
        for hr_partner in hr_partners_by_dep[parent_id]:
            if hr_partner not in hr_partners_by_dep[department_id]:
                hr_partners_by_dep[department_id].append(hr_partner)

    result = {}
    for department_id in dep_to_parent:
        if department_id not in visited_departments:
            process_department(department_id)
        result[department_id] = {
            'chiefs': chiefs_by_dep[department_id],
            'hr_partners': hr_partners_by_dep[department_id],
        }

    return result
