from django.db.models import Q

from staff.headcounts.permissions import Permissions
from staff.person.models import Staff

from staff.departments.models.department import DepartmentRoles, DepartmentStaff


class VacancyPermissions:
    def __init__(self, person: Staff):
        assert person
        self._person = person
        self._has_special_rights = None

    def vacancy_filter_qs(self) -> Q:
        if self.has_special_rights():
            all_q = ~Q(id=None)  # Filters out nothing, we can't use Q() here as django optimizes it
            return all_q

        available_budget_position_assignments = Permissions(self._person).available_budget_position_assignments()
        codes_qs = available_budget_position_assignments.values_list('budget_position__code', flat=True).distinct()
        return Q(vacancymember__person=self._person) | Q(headcount_position_code__in=codes_qs)

    def has_special_rights(self) -> bool:
        if self._has_special_rights is None:
            special_roles = (DepartmentRoles.HR_ANALYST.value, DepartmentRoles.GENERAL_DIRECTOR.value)
            has_special_role = (
                DepartmentStaff.objects
                .filter(role_id__in=special_roles, staff=self._person)
                .exists()
            )
            self._has_special_rights = self._person.user.is_superuser or has_special_role

        return self._has_special_rights
