from django.db import models
from django.db.models.signals import post_save, post_delete
from django.conf import settings
from django.core import serializers

from staff.departments.models import DepartmentStaff
from staff.lib.utils.ordered_choices import StaffChoices


class ACTION_CHOICES(StaffChoices):
    ACTION_CREATE = ('c', 'create')
    ACTION_UPDATE = ('s', 'update')
    ACTION_DELETE = ('d', 'delete')


class Log(models.Model):
    """Модель для логгирования сохранения/обновления/удаления моделей

    Логгируются только модели, перечисленные в settings.INTRANET_MODELS_LOG.
    При создании модели создается запись с action='c', при изменении создается
    запись с action='s' (раньше не было `c` и `s` значил save), при удалении
    находится существующая запись с `s` и заменяется на action='d'.

    Запись в лог происходит по сигналам

    """
    model_name = models.CharField(max_length=50, db_index=True)
    primary_key = models.IntegerField(db_index=True)

    action = models.CharField(max_length=1, choices=ACTION_CHOICES.choices())
    data = models.TextField()
    modified_at = models.DateTimeField(db_index=True, auto_now=True)

    class Meta:
        db_table = 'babylon_log'
        unique_together = (
            'model_name',
            'primary_key',
            'action',
        )
        index_together = (
            ('model_name', 'modified_at'),
        )


def start_logging():
    disabled = getattr(settings, 'DISABLE_BABYLON_LOG', False)
    if disabled:
        return

    post_save.connect(
        models_create_logger,
        dispatch_uid='babylon_models_create_logger',
    )
    post_save.connect(
        models_update_logger,
        dispatch_uid='babylon_models_update_logger',
    )
    post_delete.connect(
        models_delete_logger,
        dispatch_uid='babylon_models_delete_logger',
    )


def stop_logging():
    post_save.disconnect(
        models_create_logger,
        dispatch_uid='babylon_models_create_logger',
    )
    post_save.disconnect(
        models_update_logger,
        dispatch_uid='babylon_models_update_logger',
    )
    post_delete.disconnect(
        models_delete_logger,
        dispatch_uid='babylon_models_delete_logger',
    )


def models_create_logger(sender, *args, **kwargs):
    """Логгирование создания модели"""

    # только модели django_intranet_stuff, помеченные для логгирования
    if not (sender._meta.app_label == 'django_intranet_stuff' and
            sender.__name__ in settings.INTRANET_MODELS_LOG):
        return

    if sender is DepartmentStaff and len(kwargs['instance'].role_id) > 1:
        return

    # логгируем только созданные модели
    if not kwargs['created']:
        return

    Log.objects.create(
        model_name=sender.__name__,
        primary_key=kwargs['instance'].id,
        action=ACTION_CHOICES.ACTION_CREATE,
        data=serializers.serialize('json', [kwargs['instance']]),
    )


def models_update_logger(sender, *args, **kwargs):
    """Логгирование изменения модели"""

    # только модели django_intranet_stuff, помеченные для логгирования
    if not (sender._meta.app_label == 'django_intranet_stuff' and
            sender.__name__ in settings.INTRANET_MODELS_LOG):
        return

    if sender is DepartmentStaff and len(kwargs['instance'].role_id) > 1:
        return

    # логгируем только измененные модели
    if kwargs['created']:
        return

    # если уже есть запись про изменения этого объекта, ее нужно обновить
    log = Log.objects.get_or_create(
        model_name=sender.__name__,
        primary_key=kwargs['instance'].id,
        action=ACTION_CHOICES.ACTION_UPDATE
    )[0]
    log.data = serializers.serialize('json', [kwargs['instance']])
    log.save()


def models_delete_logger(sender, *args, **kwargs):
    """Логгирование удаления модели"""

    # только модели django_intranet_stuff, помеченные для логгирования
    # TODO: разобраться почему раньше не было проверки первого условия
    if not (sender._meta.app_label == 'django_intranet_stuff' and
            sender.__name__ in settings.INTRANET_MODELS_LOG):
        return

    # если существует запись про изменения этого объекта, ее нужно изменить
    # TODO: можно созадвать новую запись для удаления, но пока рабтает так
    try:
        log = Log.objects.get(
            model_name=sender.__name__,
            primary_key=kwargs['instance'].id,
            action=ACTION_CHOICES.ACTION_UPDATE
        )
    except Log.DoesNotExist:
        log = Log(model_name=sender.__name__, primary_key=kwargs['instance'].id)

    log.action = ACTION_CHOICES.ACTION_DELETE

    # если в списке удалябельных моделей, то записываем данные
    # TODO: разобраться почему для связок останется запись в логе,
    # а для других нет
    if sender.__name__ in getattr(settings, 'INTRANET_MODELS_LOG_DELETED', []):
        log.data = serializers.serialize('json', [kwargs['instance']])
    else:
        log.data = ''
    log.save()


start_logging()
