from django.contrib import admin

from staff.person.models import Staff

from staff.dismissal.models import MaillistDuty, CheckPoint
from staff.dismissal.admin.forms import DismissalAdminForm, CheckPointAdminForm
from staff.dismissal.admin.forms import MaillistDutyAdminForm
from staff.lib.utils.admin import StaffModelAdmin


# inlines
class MaillistDutyInline(admin.TabularInline):
    model = MaillistDuty
    form = MaillistDutyAdminForm


class CheckPointInline(admin.TabularInline):
    model = CheckPoint
    form = CheckPointAdminForm

    fields = ('name', 'name_en', 'description', 'description_en',
              'template', 'position', 'checked', 'checked_at', 'checked_by')

    readonly_fields = ('name', 'name_en', 'description', 'description_en',
                       'template', 'position', 'checked_at', 'checked_by')


# models
class DismissalAdmin(StaffModelAdmin):
    """Заявка на увольнение"""

    form = DismissalAdminForm

    fieldsets = (
        (None, {
            'fields': ('staff', 'quit_date', 'intranet_status', 'note', 'status',
                       'department', 'office')
        }),
        ('Что делать с почтой', {
            'fields': ('delete_email_address', 'delete_correspondence',
                       'forward_correspondence_to', 'give_correspondence_to')
        }),
        ('Что делать с файлами', {
            'fields': ('delete_files', 'keep_files', 'give_files_to')
        }),
        ('Метаданные', {
            'fields': ('created_by', 'created_at', 'modified_at',
                       'quit_datetime_real', 'completed_by')
        }),
    )
    inlines = (MaillistDutyInline, CheckPointInline)

    date_hierarchy = 'quit_date'
    list_display = ('staff', 'quit_date', 'department', 'office', 'status')
    list_filter = ('status', )
    search_fields = ('staff__last_name', 'staff__first_name', 'note')
    readonly_fields = ('intranet_status', 'created_at', 'modified_at',
                       'quit_datetime_real')


class ClearanceChitTemplateAdmin(admin.ModelAdmin):
    """Шаблоны обходных"""

    list_display = ('__str__', 'checkpoints_list')

    def checkpoints_list(self, obj):
        return ', '.join(obj.checkpoints.values_list('name', flat=True))


class CheckPointTemplateAdmin(admin.ModelAdmin):
    """Шаблоны пунктов обходного"""

    fieldsets = (
        ('Описание', {
            'fields': ('position', 'name', 'name_en',
                       'description', 'description_en')
        }),
        ('Ответственные', {
            'fields': ('responsibles',)
        }),
    )

    list_display = ('position', 'name', 'description')
    list_display_links = ('name',)
    list_editable = ('position',)
    raw_id_fields = ('responsibles',)

    def formfield_for_manytomany(self, db_field, request, **kwargs):
        if db_field.name == 'responsibles':
            kwargs['queryset'] = Staff.objects.filter(is_dismissed=False).order_by('department')

        return super(CheckPointTemplateAdmin, self).formfield_for_manytomany(db_field, request, **kwargs)
