from django.conf import settings
from django.shortcuts import render
from django.http import HttpResponseForbidden

from staff.person.models import Staff

from staff.dismissal.models import ClearanceChitTemplate, CheckPoint, Dismissal
from staff.dismissal.permissions import DismissalPerms as Perms
from staff.dismissal.services import DismissalService as Srv
from staff.dismissal.objects import DismissalCtl, DismissalWrapper
from staff.dismissal.debug.forms import (
    ChitTplForm, CheckpointPermsForm,
    DismissalPermsForm, DismissalCreatingPermsForm
)


def admins_only(function):
    """Декоратор, проверяет наличие имейла вошедшего в settings.ADMINS"""
    def wrapper(request, *args, **kwargs):
        authorized_emails = [a[1] for a in settings.ADMINS]
        if request.user.staff.email not in authorized_emails:
            return HttpResponseForbidden()
        return function(request, *args, **kwargs)
    return wrapper


@admins_only
def index(request):
    return render(request, 'dismissal/debug/index.html')


@admins_only
def chit(request):

    templates = ClearanceChitTemplate.objects.all()

    form = ChitTplForm(request.GET or None)
    if form.is_valid():
        staff = Staff.objects.get(login=form.cleaned_data['login'])
        staff_genitive = staff.inflections.inflect('genitive')
        exists_dismissal_procedure_for = Srv.exists_dismissal_procedure_for(staff)
        chosen_tpl = Srv.get_chit_template_for(staff)

    return render(request, 'dismissal/debug/chit.html', locals())


@admins_only
def checkpoint_perms(request):

    form = CheckpointPermsForm(request.GET or None)
    if form.is_valid():
        staff = Staff.objects.get(login=form.cleaned_data['login'])
        checkpoints_under_control = staff.checkpoint_templates.all()

        if form.cleaned_data['checkpoint_id']:
            checkpoint = CheckPoint.objects.get(pk=form.cleaned_data['checkpoint_id'])
            can_check = Perms.can_check(staff, checkpoint)

    return render(request, 'dismissal/debug/checkpoints.html', locals())


@admins_only
def dismissals(request):

    all_dismissals = Dismissal.objects.filter(intranet_status=1).select_related(
        'department', 'office', 'staff')

    form = DismissalPermsForm(request.GET or None)
    if form.is_valid():
        staff = Staff.objects.get(login=form.cleaned_data['login'])
        staff_genitive = staff.inflections.inflect('genitive')
        available_dismissals = Dismissal.objects.available_for(staff)
        available_dismissals = available_dismissals.filter(intranet_status=1)
        available_dismissals = available_dismissals.select_related(
                                             'department', 'office', 'staff')

        if form.cleaned_data['dismissal_id']:
            dismissal = Dismissal.objects.get(pk=form.cleaned_data['dismissal_id'])
            dismissal = DismissalWrapper(dismissal)
            can_update_dismissal = Perms.can_update_dismissal(staff, dismissal)
            can_cancel_dismissal = Perms.can_cancel_dismissal(staff, dismissal)

    return render(request, 'dismissal/debug/dismissals.html', locals())


@admins_only
def creating(request):

    form = DismissalCreatingPermsForm(request.GET or None)
    if form.is_valid():
        who = Staff.objects.get(login=form.cleaned_data['who_login'])
        who_genitive = who.inflections.inflect('genitive')
        for_whom = Staff.objects.get(login=form.cleaned_data['for_whom_login'])
        for_whom_dative = for_whom.inflections.inflect('dative')
        for_whom_genitive = for_whom.inflections.inflect('genitive')

        exists_dismissal_procedure_for = Srv.exists_dismissal_procedure_for(for_whom)
        can_see_dismissal_link = DismissalCtl().can_see_dismissal_link(who, for_whom)
        can_create_dismissal = Perms.can_create_dismissal(who, for_whom)
        can_dismiss_from_anketa = Perms.can_dismiss_from_anketa(who, for_whom)
        can_dismiss_from_chit = Perms.can_dismiss_from_chit(who, for_whom)
        can_see_hr_note = Perms.can_see_hr_note(who, for_whom)

    return render(request, 'dismissal/debug/creating.html', locals())
