from django.dispatch.dispatcher import receiver

from staff.lib.models.departments_chain import DepartmentsChain

from staff.dismissal.services import DismissalService
from staff.dismissal.signals import (
    dismissal_created,
    dismissal_updated,
    dismissal_cancelled,
    dismissal_completed,
    dismissal_quit_date,
    dismissal_not_completed,
    dismissal_date_passed
)
from staff.dismissal.notifications import (
    DismissalNotificationContext as Context,
    CompletionSuperiorNotification,
    CompletionNotification,
    CreationNotification,
    UpdateNotification,
    CancellationNotification,
    QuitDateNotification,
    NotCompletedNotification,
    DatePassedNotification,
    MailFilesActionsNotification,
    MailSearchDeletionNotification
)

from .utils import get_chief_for_dismissal

import logging
logger = logging.getLogger('dismissal')


def make_sender(person):
    return (
        '%s %s' % (person.first_name, person.last_name),
        '%s@yandex-team.ru' % person.login,
    )


@receiver(dismissal_created)
def dismissal_created_notify(dismissal, **kwargs):
    """
    Уведомляем непосредственного начальника, если не он создал увольнение,
    либо тикет для hr
    """
    direct_superior = dismissal.staff.get_superior()
    context = Context(
        staff=dismissal.staff,
        dismissal=dismissal,
    )

    CreationNotification(
        context=context,
        target='DISMISSAL_ANY',
        department=dismissal.department,
        office=dismissal.office,
        staff=dismissal.staff
    ).send()

    if direct_superior and dismissal.created_by != direct_superior:
        CreationNotification(
            context=context,
            target='@',
            department=dismissal.department,
            office=dismissal.office,
            staff=dismissal.staff
        ).send(recipients=[direct_superior.work_email])

    CreationNotification(
        context=context,
        target='DISMISSAL_HR',
        department=dismissal.department,
        office=dismissal.office,
        staff=dismissal.staff
    ).send(sender=make_sender(dismissal.created_by))


@receiver(dismissal_updated)
def dismissal_updated_notify(dismissal, updated_by, changes, **kwargs):
    """Сообщаем об измененениях руководителю"""
    direct_superior = dismissal.staff.get_superior()
    changes = {
        k: v for k, v in changes.items()
        if k not in dismissal.NEW_FIELDS
    }
    context = Context(
        staff=dismissal.staff,
        dismissal=dismissal,
        staff_name_genitive=dismissal.staff.inflections.inflect('genitive'),
        updated_by=updated_by,
        changes=DismissalService.humanize_dismissal_changes(changes),
    )
    need_notify_superior = (
        direct_superior and
        updated_by != direct_superior and
        not (set(dismissal.NEW_FIELDS) & set(changes))
    )
    if need_notify_superior:
        UpdateNotification(
            context=context,
            target='@',
            department=dismissal.department,
            office=dismissal.office,
            staff=dismissal.staff
        ).send(recipients=[direct_superior.work_email])


@receiver(dismissal_updated)
def dismissal_updated_hr_notify(dismissal, updated_by, changes, **kwargs):
    """Сообщаем об измененениях hr"""

    # им интересна дата и комментарий
    # Костыли вы мои костыли...
    if ('quit_date' not in changes and 'note' not in changes and
            all(f not in changes for f in dismissal.HR_FIELDS)):
        return
    hr_related_changes = {}
    if 'quit_date' in changes:
        hr_related_changes['quit_date'] = changes['quit_date']
    if 'note' in changes:
        hr_related_changes['note'] = changes['note']
    for field in dismissal.HR_FIELDS:
        if field in changes:
            hr_related_changes[field] = changes[field]

    context = Context(
        staff=dismissal.staff,
        dismissal=dismissal,
        staff_name_genitive=dismissal.staff.inflections.inflect('genitive'),
        updated_by=updated_by,
        changes=DismissalService.humanize_dismissal_changes(hr_related_changes),
    )
    UpdateNotification(
        context=context,
        target='DISMISSAL_HR',
        department=dismissal.department,
        office=dismissal.office,
        staff=dismissal.staff
    ).send(sender=make_sender(updated_by))


@receiver(dismissal_cancelled)
def dismissal_cancelled_notify(dismissal, cancelled_by, **kwargs):
    """Сообщаем об отмене увольнения руководителю"""
    direct_superior = dismissal.staff.get_superior()

    context = Context(
        staff=dismissal.staff,
        dismissal=dismissal,
        staff_name_genitive=dismissal.staff.inflections.inflect('genitive'),
        cancelled_by=cancelled_by,
        changes={},
    )
    if direct_superior and cancelled_by != direct_superior:
        CancellationNotification(
            context=context,
            target='@',
            department=dismissal.department,
            office=dismissal.office,
            staff=dismissal.staff
        ).send(recipients=[direct_superior.work_email])


@receiver(dismissal_completed)
def dismissal_completed_superior_notify(staff, **kwargs):
    """
    Уведомляем руководитя об увольнении сотрудника `staff`
    Если передан объект Dismissal, то в письме дадим ссылку на него вместо вики
    """
    context = Context(
        staff=staff,
        dismissal=kwargs.get('dismissal'),
    )

    direct_superior = staff.get_superior()
    if direct_superior:
        CompletionSuperiorNotification(
            context=context,
            target='@',
            department=staff.department,
            office=staff.office
        ).send(recipients=[direct_superior.work_email])


@receiver(dismissal_completed)
def dismissal_completed_broadcast_notify(staff, chiefed_departments,
                                         deputed_departments, **kwargs):
    """Уведомляем заинтересованных об увольнении сотрудника `staff`"""

    dch = kwargs['departments_chains']
    departments_chain = dch.get_chain_as_list(department=staff.department)
    chiefed_chains = [dch.get_chain_as_list(department=d)
                      for d in chiefed_departments]
    deputed_chains = [dch.get_chain_as_list(department=d)
                      for d in deputed_departments]

    context = Context(
        staff=staff,
        dismissal=kwargs.get('dismissal'),
        departments_chain=departments_chain,
        chiefed_chains=chiefed_chains,
        deputed_chains=deputed_chains,
    )
    CompletionNotification(
        context=context,
        target='DISMISSAL_BROADCAST',
        department=staff.department,
        office=staff.office
    ).send()


@receiver(dismissal_completed)
def dismissal_mail_files_notify(staff, **kwargs):
    """Уведомляем о том что делать с почтой и файлами"""

    dismissal = kwargs.get('dismissal')
    if dismissal is None or not dismissal.mail_and_files_completed:
        return

    dch = kwargs['departments_chains']
    departments_chain = dch.get_chain_as_list(department=staff.department)

    MailFilesActionsNotification(
        context=Context(
            staff=staff,
            dismissal=dismissal,
            departments_chain=departments_chain,
        ),
        target='DISMISSAL_MAIL_FILES',
        department=staff.department,
        office=staff.office
    ).send()


@receiver(dismissal_completed)
def dismissal_mail_search_files_notify(staff, **kwargs):
    """Уведомляем о том что делать с файлами поиска"""

    dismissal = kwargs.get('dismissal')
    if dismissal is None:
        return
    if not (dismissal.delete_from_search or dismissal.comment):
        return

    dch = kwargs['departments_chains']
    departments_chain = dch.get_chain_as_list(department=staff.department)

    MailSearchDeletionNotification(
        context=Context(
            staff=dismissal.staff,
            dismissal=dismissal,
            chief=get_chief_for_dismissal(dismissal),
            departments_chain=departments_chain,
        ),
        target='DISMISSAL_SEEK_ACCESS',
        department=staff.department,
        office=staff.office,
    ).send()


@receiver(dismissal_updated)
def dismissal_mail_files_after_dismissal_notify(dismissal, updated_by, changes, **kwargs):
    """
    Уведомление про обновление информации про почту и файлы после увольнения
    """
    if not dismissal.is_completed:
        return

    dch = DepartmentsChain()
    departments_chain = dch.get_chain_as_list(
        department=dismissal.staff.department)

    MailFilesActionsNotification(
        context=Context(
            staff=dismissal.staff,
            dismissal=dismissal,
            updated_by=updated_by,
            updated_by_ablative=updated_by.inflections.inflect('ablative'),
            departments_chain=departments_chain,
        ),
        target='DISMISSAL_MAIL_FILES',
        department=dismissal.staff.department,
        office=dismissal.staff.office
    ).send()

    if any(f in changes for f in dismissal.SEARCH_FIELDS):
        MailSearchDeletionNotification(
            context=Context(
                staff=dismissal.staff,
                dismissal=dismissal,
                chief=get_chief_for_dismissal(dismissal),
                departments_chain=departments_chain,
            ),
            target='DISMISSAL_SEEK_ACCESS',
            department=dismissal.staff.department,
            office=dismissal.staff.office,
        ).send()


@receiver(dismissal_quit_date)
def dismissal_quit_date_notify(dismissal, **kwargs):
    """
    Письмо ответственным в день увольнения сотрудника, отсылаем только про
    неотмеченные пункты.
    """

    unchecked_checkpoints = dismissal.checkpoints.filter(checked=False)
    for checkpoint in unchecked_checkpoints:
        if checkpoint.template is None:
            continue
        recipients = checkpoint.template.responsibles.values_list(
            'work_email', flat=True)
        QuitDateNotification(
            context=Context(
                staff=dismissal.staff,
                dismissal=dismissal,
                checkpoint=checkpoint,
            ),
            target='@',
            department=dismissal.department,
            office=dismissal.office,
            staff=dismissal.staff
        ).send(recipients=recipients)


@receiver(dismissal_not_completed)
def dismissal_not_completed_notify(dismissal, **kwargs):
    """
    Письмо ответственным, что пункт не заполнен
    """
    for checkpoint in dismissal.checkpoints.filter(checked=False):
        if checkpoint.template is None:
            continue
        recipients = checkpoint.template.responsibles.values_list(
            'work_email', flat=True)
        NotCompletedNotification(
            context=Context(
                staff=dismissal.staff,
                dismissal=dismissal,
                checkpoint=checkpoint,
            ),
            target='@',
            department=dismissal.department,
            office=dismissal.office,
            staff=dismissal.staff
        ).send(recipients=recipients)


@receiver(dismissal_date_passed)
def dismissal_date_passed_notify(dismissal, **kwargs):
    """
    Cотрудник не уволен, хотя дата прошла
    """
    DatePassedNotification(
        context=Context(
            staff=dismissal.staff,
            dismissal=dismissal,
        ),
        target='DISMISSAL_HR',
        department=dismissal.department,
        office=dismissal.office,
        staff=dismissal.staff
    ).send()
