from staff.reports.sources import get_chiefs, get_persons

from staff.reports.utils import full_name_wrapper
from staff.reports.sources import (
    get_offices,
    get_departments,
    get_oebs_full_names,
)
from staff.reports.utils import populate_departments
from staff.reports.objects.base import Report

from ..models import DISMISSAL_STATUS
from ..queries import get_dismissals


class DismissalReport(Report):
    queryset = get_dismissals
    sheet_name = 'yandex_dismissal'

    def get_head_rows(self):
        yield [
            'ФИО',
            'ФИО по данным из Оебс',
            'Логин',
            'Дата найма',
            'Цепочка подразделений',
            'Офис',
            'Логин руководителя',
            'Дата создания заявки на увольнение',
            'Логин сотрудника, который создал заявку',
            'Цепочка подразделений сотрудника, который создал заявку',
            'Логин сотрудника, который уволил',
            'Дата увольнения в заявке',
            'Дата увольнения сотрудника',
            'Уволен',
            'Статус заявки',
            'Инициатор',
            'Нужна помощь HR',
            'Комментарий для HR',
            'Причина ухода',
            'Новое место работы',
            'Отношение к уходу',
            'Удалить емейл',
            'Удалить почту',
            'Кому перенаправлять почту',
            'Кому архивировать и передать файлы (логин)',
            'Рекомендуем к повторному найму',
        ]

    @staticmethod
    def get_status_presentation(status):
        status_text = {
            DISMISSAL_STATUS.IN_PROGRESS.lower(): 'В процессе',
            DISMISSAL_STATUS.CANCELLED.lower(): 'Заявка отменена',
            DISMISSAL_STATUS.DATE_PASSED.lower(): 'Hе переведен в бывшие',
            DISMISSAL_STATUS.CHIT_NOT_COMPLETE.lower(): 'Обходной не заполнен',
            DISMISSAL_STATUS.DONE.lower(): 'Уволен',
            DISMISSAL_STATUS.NEW.lower(): 'Новая',
        }
        return status_text[status.lower()]

    def join_data(self):
        person_fields = [
            'id',
            'department',
            'login',
            'first_name',
            'middle_name',
            'last_name',
            'join_at',
            'quit_at',
            'is_dismissed',
        ]
        persons = {
            p['id']: p
            for p in get_persons().values(*person_fields)
        }
        offices = {
            o['id']: o
            for o in get_offices()
        }

        chiefs = get_chiefs()
        departments_qs = get_departments()
        departments = populate_departments(departments_qs, chiefs)
        oebs_full_names = get_oebs_full_names()

        field_getters = [
            ('staff', lambda p_id: persons[p_id]),
            ('completed_by', lambda p_id: persons[p_id]['login']),
            ('created_by', lambda p_id: persons[p_id]),
            ('department', lambda dep_id: departments[dep_id]),
            ('office', lambda o_id: offices[o_id]['name']),
            ('forward_correspondence_to', lambda p_id: persons[p_id]['login']),
            ('give_files_to', lambda p_id: persons[p_id]['login']),
        ]

        for dismissal in get_dismissals().order_by('id'):
            for field, getter in field_getters:
                if dismissal[field]:
                    dismissal[field] = getter(dismissal[field])

            dismissal['oebs_full_name'] = oebs_full_names.get(dismissal['staff']['id'], '')
            dismissal['creator_department'] = departments[
                dismissal['created_by']['department']
            ]

            dismissal['status'] = self.get_status_presentation(dismissal['status'].lower())
            yield dismissal

    def get_rows(self):
        dateformat = '%d.%m.%Y'

        def yes_no_option(val):
            return 'Да' if val else 'Нет'

        def str_date(date):
            return date.strftime(dateformat) if date else ''

        dismissals = self.join_data()

        for dismissal in dismissals:
            person = dismissal['staff']
            yield [
                full_name_wrapper(person),
                dismissal['oebs_full_name'],
                person['login'],
                person['join_at'].strftime(dateformat),
                ' => '.join(d['name']for d in dismissal['department']['bread_crumb']),
                dismissal['office'],
                dismissal['department']['chief']['login'],
                str_date(dismissal['created_at']),
                dismissal['created_by']['login'],
                ' => '.join(
                    d['name']
                    for d in dismissal['creator_department']['bread_crumb']
                ),
                dismissal['completed_by'],
                str_date(dismissal['quit_date']),
                str_date(dismissal['quit_datetime_real']),
                yes_no_option(person['is_dismissed']),
                dismissal['status'],
                dismissal['initiator'],
                yes_no_option(dismissal['need_hr_help']),
                dismissal['note'],
                dismissal['reason'],
                dismissal['new_employer'],
                dismissal['impression'],
                yes_no_option(dismissal['delete_email_address']),
                yes_no_option(dismissal['delete_correspondence']),
                dismissal['forward_correspondence_to'],
                dismissal['give_files_to'],
                dismissal['rehire_recommendation'],
            ]
