from django.db.models import Q, Count

from staff.lib.models.mptt import get_heirarchy_filter_query

from staff.dismissal.models import ClearanceChitTemplate as ChitTpl


class DismissalService(object):

    @staticmethod
    def get_chit_template_for(staff):
        """
        Среди шаблонов обходных находим наиболее подходящий исходя из
        подразделения и офиса сотрудника.
        """
        department_q = (Q(department__in=staff.departments) |
                        Q(department__isnull=True))
        office_q = Q(office=staff.office) | Q(office__isnull=True)
        chit_tpls = ChitTpl.objects.filter(department_q & office_q)

        if chit_tpls:
            return chit_tpls.order_by('department', 'office')[0]
        return None

    @classmethod
    def exists_dismissal_procedure_for(cls, staff):
        """
        Существует ли процедура увольнения с обходным для сотрудника или он
        сразу увольняется по кнопке на анкете
        """
        chosen_tpl = cls.get_chit_template_for(staff)
        if chosen_tpl:
            return chosen_tpl.checkpoints.exists()
        return False

    @classmethod
    def exists_dismissal_procedure(cls, target_data) -> bool:
        """
        Существует ли процедура увольнения с обходным для сотрудника или он
        сразу увольняется по кнопке на анкете
        """
        hf = get_heirarchy_filter_query([target_data], False, 'department__', 'department__', True)
        department_q = (hf | Q(department__isnull=True))
        office_q = Q(office_id=target_data['office__id']) | Q(office__isnull=True)
        chit_template = (
            ChitTpl.objects
            .filter(department_q & office_q)
            .values('pk')
            .annotate(checkpoints=Count('checkpoints'))
            .order_by('department', 'office')
            .first()
        )

        return bool(chit_template and chit_template['checkpoints'] > 0)

    @staticmethod
    def humanize_dismissal_changes(changes):
        from django.utils.translation import ugettext_lazy as _
        human_readable_diff = {}

        for key, val in changes.items():
            key = key.split('__id')[0]
            human_key = _('dismissal.%s-label' % key)
            if isinstance(val, bool):
                val = 'да' if val else 'нет'
            human_readable_diff[human_key] = val

        return human_readable_diff
