import logging
from functools import wraps

from staff.celery_app import app
from datetime import datetime, timedelta, date

from staff.lib.tasks import LockedTask
from staff.dismissal.models import Dismissal, DISMISSAL_STATUS
from staff.dismissal.signals import (
    dismissal_quit_date,
    dismissal_not_completed
)
from staff.dismissal.objects import DismissalCtl


logger = logging.getLogger('dismissal')


def log_command_failure(msg):
    def _func(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except Exception:
                logger.exception(msg)
        return wrapper
    return _func


@app.task()
class ClearInactiveDismissalsSignalTask(LockedTask):
    """
    Удаляем неактивные заявки за исключением отменённых
    """

    @log_command_failure('ClearInactiveDismissalsSignalTask')
    def locked_run(self, *args, **kwargs):

        days_ago = int(kwargs.get('days_ago', 30))
        inactive_dismissals = Dismissal.objects.filter(
            intranet_status=0,
            created_at__lte=date.today() - timedelta(days=days_ago)
        ).exclude(status=DISMISSAL_STATUS.CANCELLED)

        logger.info('ClearInactiveDismissalsSignalTask: %s', inactive_dismissals.values_list('pk', flat=True))
        inactive_dismissals.delete()


@app.task()
class SendChitNotCompleteSignalTask(LockedTask):

    @log_command_failure('SendChitNotCompleteSignalTask')
    def locked_run(self, *args, **kwargs):
        week_ago = datetime.now() - timedelta(days=7)

        incomplete_dismissals = Dismissal.objects.filter(
            intranet_status=1,
            status=DISMISSAL_STATUS.CHIT_NOT_COMPLETE,
            deadline__gte=week_ago,
        )

        for num, dismissal in enumerate(incomplete_dismissals, start=1):
            dismissal_not_completed.send(
                sender=object,
                dismissal=DismissalCtl(dismissal=dismissal).wrapper
            )


@app.task()
class SendQuitDateSignalTask(LockedTask):

    @log_command_failure('SendQuitDateSignalTask')
    def locked_run(self, *args, **kwargs):
        dismissals_today = Dismissal.objects.filter(
            intranet_status=1,
            status=DISMISSAL_STATUS.IN_PROGRESS,
            deadline=date.today()
        )

        for num, dismissal in enumerate(dismissals_today, start=1):
            dismissal_quit_date.send(
                sender=object,
                dismissal=DismissalCtl(dismissal=dismissal).wrapper
            )


@app.task()
class CompleteDismissalsTask(LockedTask):
    @log_command_failure('CompleteDismissalsTask')
    def locked_run(self, *args, **kwargs):
        DismissalCtl.complete_passed()
