import json
import logging

from pytz import timezone

from django.http import HttpResponseRedirect
from django.core.urlresolvers import reverse
from django.views.decorators.http import require_http_methods
from django.views.decorators.csrf import ensure_csrf_cookie

from staff.person.models import Staff

from staff.lib.forms.staff_form_grid import StaffFormGrid
from staff.lib.forms.errors import invalid_json_error
from staff.lib.utils.qs_values import localize
from staff.lib.decorators import responding_json, available_for_external
from staff.lib.paginator import paginate

from staff.dismissal.services import DismissalService
from staff.dismissal.models import Dismissal, CheckPoint, DISMISSAL_STATUS
from staff.dismissal.permissions import DismissalPerms
from staff.dismissal.objects import DismissalCtl, CheckpointCtl

from .utils import (
    get_checkpoint_dict,
    person_location,
    localize_staff,
    to_tz,
)
from .forms import DismissalForm, FilterDismissalsForm

logger = logging.getLogger('dismissal_api.views')


def _get_completed_dict(dismissal):
    return {
        'checked_at': dismissal.quit_datetime_real,
        'checked_by': {
            'first_name': dismissal.completed_by.i_first_name,
            'last_name': dismissal.completed_by.i_last_name,
            'login': dismissal.completed_by.login,
            'id': dismissal.completed_by.id,
        },
    }


@ensure_csrf_cookie
@require_http_methods(['GET'])
@responding_json
def dismissal_list(request):
    """Список заявок на увольнение"""

    requested_by = request.user.get_profile()
    form_name = 'filter_dismissals'
    REQUESTED_FIELDS = (
        'id',
        'quit_date',
        'deadline',
        'staff__first_name',
        'staff__first_name_en',
        'staff__last_name',
        'staff__last_name_en',
        'staff__login',
        'department__name',
        'department__name_en',
        'office__name',
        'office__name_en',
        'status',
    )

    grid = StaffFormGrid(FilterDismissalsForm, data=[request.GET.dict()])
    if not grid.is_valid():
        return {'errors': {form_name: grid.errors()}}

    filter_data = grid.cleaned_data[0]
    date_from, date_to = filter_data.get('date_from'), filter_data.get('date_to')
    dismissal_status = filter_data.get('status')
    requested_page = request.GET.get('page')

    dismissals = (
        Dismissal.objects
        .available_for(requested_by)
        .filter(intranet_status=1)
        .order_by('-quit_date')
        .values(*REQUESTED_FIELDS)
    )
    if date_from and date_to:
        dismissals = dismissals.filter(
            quit_date__gte=date_from, quit_date__lte=date_to)
    if dismissal_status:
        dismissals = dismissals.filter(status=dismissal_status)

    dismissals = paginate(
        object_list=dismissals,
        per_page=50,
        page=requested_page
    )

    def dismissal_dict(dismissal):
        localized = localize(dismissal)
        return {
            'id': localized['id'],
            'quit_date': localized['quit_date'],
            'first_name': localized['staff__first_name'],
            'last_name': localized['staff__last_name'],
            'login': localized['staff__login'],
            'department': localized['department__name'],
            'status': localized['status'],
            'office': localized['office__name'],
        }

    return {
        'dismissals': [
            dismissal_dict(dismissal) for dismissal in dismissals
        ],
        'choices': grid.choices_as_front_dict(),
        'page': dismissals.number,
        'pages': dismissals.paginator.num_pages,
        'selected_choices': request.GET.dict(),
    }


@ensure_csrf_cookie
@require_http_methods(['GET', 'POST'])
@available_for_external('dismissal.can_dismiss_from_anketa')
@responding_json
def create_dismissal(request, login):
    requested_by = request.user.get_profile()
    try:
        target_person = Staff.objects.get(is_dismissed=False, login=login)
    except Staff.DoesNotExist:
        return {'errors': ['dismissal-person_does_not_exist']}, 404

    dismissal_ctl = DismissalCtl(author=requested_by)
    can_see_link = dismissal_ctl.can_see_dismissal_link(subject=requested_by, object_=target_person)
    if not can_see_link:
        return {'errors': ['permission_denied']}, 403

    can_see_hr_note = DismissalPerms.can_see_hr_note(subject=requested_by, object_=target_person)
    can_set_deadline = DismissalPerms.can_set_deadline(subject=requested_by)

    if request.method == 'GET':
        if not DismissalService.exists_dismissal_procedure_for(target_person):
            return {'errors': ['dismissal-procedure-does-not-exist']}, 400

        dismissal_form = DismissalForm(initial={
            'can_set_deadline': can_set_deadline,
            'can_see_hr_note': can_see_hr_note,
            'is_completed': False,
        })

        dismissal_data = {
            'structure': dismissal_form.structure_as_dict(),
            'dismissal_staff': localize_staff(target_person),
            'location': person_location(target_person),
        }

        return dismissal_data

    # POST
    dismiss_fast = not DismissalService.exists_dismissal_procedure_for(target_person)
    if dismiss_fast:
        DismissalCtl(author=requested_by, person=target_person).complete_fast()
        return HttpResponseRedirect(
            reverse('anketa-anketa', kwargs={'yandex_login': login})
        )

    if not DismissalPerms.can_create_dismissal(subject=requested_by, object_=target_person):
        return {'errors': ['permission_denied']}, 403

    try:
        req_data = json.loads(request.body)
    except ValueError:
        logger.exception('Wrong JSON: %s', request.body)
        return invalid_json_error(request.body), 400

    dismissal_form = DismissalForm(data=req_data, initial={
        'can_set_deadline': can_set_deadline,
        'can_see_hr_note': can_see_hr_note,
        'is_completed': False,
    })

    if dismissal_form.is_valid():
        dismissal = dismissal_ctl.get_or_create(staff=target_person)
        dismissal_ctl = DismissalCtl(dismissal=dismissal, author=requested_by)
        dismissal_ctl.update(dismissal_form.cleaned_data, enable=True)
        return {'dismissal_id': dismissal.id}
    else:
        return dismissal_form.errors


def _get_dismissal_obj(dismissal_id, requested_by):
    dismissal = (
        Dismissal.objects
        .available_for(requested_by)
        .exclude(status=DISMISSAL_STATUS.CANCELLED)
        .select_related()
        .get(id=dismissal_id)
    )
    return dismissal


@ensure_csrf_cookie
@require_http_methods(['GET'])
@responding_json
def get_dismissal(request, dismissal_id):

    requested_by = request.user.get_profile()
    observer_tz = timezone(requested_by.tz)

    try:
        dismissal = _get_dismissal_obj(dismissal_id, requested_by)
    except Dismissal.DoesNotExist:
        return {'errors': ['dismissal-dismissal_does_not_exist']}, 404

    dismissal_ctl = DismissalCtl(dismissal=dismissal, author=requested_by)

    can_see_hr_note = DismissalPerms.can_see_hr_note(subject=requested_by, object_=dismissal.staff)
    can_set_deadline = DismissalPerms.can_set_deadline(subject=requested_by)
    can_dismiss_from_chit = DismissalPerms.can_dismiss_from_chit(subject=requested_by, object_=dismissal.staff)
    is_completed = dismissal_ctl.wrapper.is_completed

    dismissal_form = DismissalForm(initial={
        'dismissal': dismissal,
        'can_set_deadline': can_set_deadline,
        'can_see_hr_note': can_see_hr_note,
        'is_completed': is_completed,
    })

    dismissal_data = {
        'id': dismissal_id,
        'is_active': dismissal_ctl.wrapper.is_active,
        'is_completed': is_completed,
        'can_dismiss_from_chit': can_dismiss_from_chit,
        'dismissal_staff': localize_staff(dismissal.staff),
        'location': person_location(dismissal.staff),
        'checkpoints': [
            get_checkpoint_dict(checkpoint, requested_by)
            for checkpoint in dismissal_ctl.wrapper.checkpoints
        ],
    }
    dismissal_data.update(dismissal_form.as_dict())

    if dismissal_data['is_completed']:
        dismissal_data['completed'] = _get_completed_dict(dismissal)

    return to_tz(dismissal_data, observer_tz)


@require_http_methods(['POST'])
@responding_json
def edit_dismissal(request, dismissal_id):
    requested_by = request.user.staff

    try:
        dismissal = _get_dismissal_obj(dismissal_id, requested_by)
    except Dismissal.DoesNotExist:
        return {'errors': ['dismissal-dismissal_does_not_exist']}, 404

    if not DismissalPerms.can_create_dismissal(subject=requested_by, object_=dismissal.staff):
        return {'errors': ['permission_denied']}, 403

    dismissal_ctl = DismissalCtl(dismissal=dismissal, author=requested_by)

    try:
        req_data = json.loads(request.body)
    except ValueError:
        logger.exception('Wrong JSON: %s', request.body)
        return invalid_json_error(request.body), 400

    can_see_hr_note = DismissalPerms.can_see_hr_note(subject=requested_by, object_=dismissal.staff)
    can_set_deadline = DismissalPerms.can_set_deadline(subject=requested_by)

    dismissal_form = DismissalForm(data=req_data, initial={
        'can_set_deadline': can_set_deadline,
        'can_see_hr_note': can_see_hr_note,
        'is_completed': dismissal_ctl.wrapper.is_completed,
    })

    if not dismissal_form.is_valid():
        return dismissal_form.errors

    dismissal_ctl.update(dismissal_form.cleaned_data)
    return {'dismissal_id': dismissal.id}


@require_http_methods(['POST'])
@responding_json
def cancel_dismissal(request, dismissal_id):
    requested_by = request.user.staff

    try:
        dismissal = _get_dismissal_obj(dismissal_id, requested_by)
    except Dismissal.DoesNotExist:
        return {'errors': ['dismissal-dismissal_does_not_exist']}, 404

    dismissal_ctl = DismissalCtl(dismissal=dismissal, author=requested_by)

    if not DismissalPerms.can_cancel_dismissal(subject=requested_by, dismissal=dismissal):
        return {'errors': ['permission_denied']}, 403
    if dismissal_ctl.can_be_cancelled():
        dismissal_ctl.cancel()
    return {}


@require_http_methods(['POST'])
@responding_json
def complete_dismissal(request, dismissal_id):
    requested_by = request.user.get_profile()
    try:
        dismissal = (
            Dismissal.objects
            .get(id=dismissal_id)
        )
    except Dismissal.DoesNotExist:
        return {'errors': ['dismissal-dismissal_does_not_exist']}, 404

    if dismissal.status in (DISMISSAL_STATUS.DONE, DISMISSAL_STATUS.CANCELLED):
        return {'status': 'already dismissed'}

    if not DismissalPerms.can_dismiss_from_chit(subject=requested_by, object_=dismissal.staff):
        return {'errors': ['permission_denied']}, 403

    try:
        DismissalCtl(dismissal=dismissal, author=requested_by).complete(by_whom=requested_by)
    except DismissalCtl.PersonNotMoveToExt:
        return {'errors': ['person_not_move_to_ext']}, 400

    dismissal = Dismissal.objects.get(pk=dismissal.pk)

    observer_tz = timezone(requested_by.tz)
    return to_tz(_get_completed_dict(dismissal), tz=observer_tz)


@require_http_methods(['POST'])
@responding_json
def confirm_checkpoint(request, checkpoint_id):

    try:
        checkpoint = CheckPoint.objects.get(id=checkpoint_id)
    except CheckPoint.DoesNotExist:
        return {"errors": ["dismissal-checkpoint_does_not_exist"]}, 404

    requested_by = request.user.get_profile()

    if checkpoint.checked_at:
        return {'errors': ['Already checked']}

    if not DismissalPerms.can_check(subject=requested_by, checkpoint=checkpoint):
        return {"errors": ["permission_denied"]}, 403

    cp_ctl = CheckpointCtl(author=requested_by, checkpoint=checkpoint)
    cp_ctl.check(checkpoint=checkpoint)

    observer_tz = timezone(requested_by.tz)
    return to_tz(get_checkpoint_dict(checkpoint, requested_by), tz=observer_tz)
