from functools import wraps
from django.http import HttpRequest, HttpResponseForbidden

from .checkers import (
    token_is_valid,
    referer_is_valid,
    is_oauth,
    is_authenticated,
)

__all__ = ('token_auth', 'check_referer', 'passport_or_token_auth')


def _check_all(referer=False, passport=False, token=False):

    def _check(args):

        request = None
        for arg in args:
            if isinstance(arg, HttpRequest):
                request = arg
        if not request:
            return 'Wrong HttpRequest'

        access = True
        if token:
            access = token_is_valid(request)
            if access:
                return ''

        if passport:
            access = is_authenticated(request)

        if referer:
            access = access and (
                referer_is_valid(request) or is_oauth(request)
            )

        if access:
            return ''

        messages = [
            [token, 'Invalid token or unallowed host'],
            [passport, 'User not authorized'],
            [referer, 'Invalid or empty referer'],
        ]

        return ' or '.join(msg for e, msg in messages if e)

    def decorator(func):

        @wraps(func)
        def wrapper(*args, **kwargs):
            msg = _check(args)
            if msg:
                return HttpResponseForbidden(
                    'Forbidden: %s' % msg,
                    content_type='text/plain',
                )
            return func(*args, **kwargs)

        return wrapper

    return decorator


token_auth = _check_all(referer=False, passport=False, token=True)
token_auth.__doc__ = """
    Decorator for checking auth token
    There is HttpRequest object must be
    in arguments of decorated function or class method
"""

check_referer = _check_all(referer=True, passport=False, token=False)
check_referer.__doc__ = """
    Decorator for checking referer to prevent CSRF
    If there is GET variable "token" in request checking will be skipped
    HttpRequest object must be in arguments
    of decorated function or class method
"""


def passport_or_token_auth(check_referer=True):
    """
    Decorator for checking referer to prevent CSRF (if check_referer is True)

    If there is GET variable "token" in request checking will be skipped
    There is HttpRequest object must be in arguments
    of decorated function or class method
    """

    if check_referer:
        checker = _check_all(referer=True, passport=True, token=True)
        # hack to be backward-compatible
        # (to use @passport_or_token_auth instead of
        #  @passport_or_token_auth())
        if callable(check_referer):
            return checker(check_referer)
        else:
            return checker
    else:
        return _check_all(referer=False, passport=True, token=True)
