import re
from django.db import models
from staff.django_api_auth.utils import drop_cache_token

re_delim = re.compile(r'[\s,;\|]+')


class Token(models.Model):
    token = models.CharField(max_length=40, db_index=True)
    hostnames = models.TextField(null=True, blank=True)
    ips = models.TextField(null=True, blank=True)
    description = models.TextField(null=True)
    created_at = models.DateTimeField(auto_now_add=True)
    modified_at = models.DateTimeField(auto_now=True, db_index=True)

    def __str__(self):
        return self.token

    def save(self, *args, **kwargs):
        if self.id:
            try:
                old_state = Token.objects.get(id=self.id)
                drop_cache_token(
                    token=old_state.token,
                    ips=old_state.ips_list,
                    hostnames=old_state.hostnames_list,
                )
            except Token.DoesNotExist:
                pass
        r = super(Token, self).save(*args, **kwargs)
        drop_cache_token(
            token=self.token,
            ips=self.ips_list,
            hostnames=self.hostnames_list,
        )
        return r

    @property
    def ips_list(self):
        return self._split_values(self.ips) if self.ips else []

    @property
    def hostnames_list(self):
        return self._split_values(self.hostnames) if self.hostnames else []

    @classmethod
    def _split_values(cls, values):
        return list(set(re_delim.sub(' ', values).lower().strip().split()))
