import logging
import re
import yenv

from django.conf import settings

from .messages import Message, MessageThread
from .utils.wiki import WikiFormatter
from .backends import MessageWrapper

logger = logging.getLogger(__name__)


class EmailMessageThread(MessageThread):
    transport_id = 'email'

    def increment(self, msg_id):
        if msg_id is None:
            id = 0
        else:
            try:
                id = int(
                    re.match(r'\S+\.\S+\.(\d+)@.*', msg_id).groups()[0]) + 1
            except AttributeError:
                id = 0

        return '<%s.%s.%s@%s>' % (self.subj_id, self.target,
                                  id, settings.EMAIL_DOMAIN)


class HTMLEmailMessageThread(EmailMessageThread):
    transport_id = 'html_email'


class HTMLEmailMessage(Message):
    template = None

    def prepare_headers(self, recipients, sender):
        recipients = list(recipients)
        recipients += settings.EMAIL_COPY

        emails = []
        for email in recipients:
            if not email:
                logger.error('Empty email in recipients list: %s', recipients)
            elif email[-1] == '@':
                emails.append(email + 'yandex-team.ru')
            else:
                emails.append(email)

        if yenv.type != 'production':
            self.context['original_emails'] = emails
            emails = [settings.DEBUG_EMAIL]
        self.recipients = emails
        if not sender:
            sender = settings.DEFAULT_FROM_EMAIL
        return emails, sender

    def send(self, **params):
        recipients = params.get('recipients')
        if not recipients:
            return

        sender = params.get('sender')

        recipients, sender = self.prepare_headers(recipients, sender)

        logger.info("Sending notification mail to <<%s>> from %s,"
                    "with message_id: '%s',"
                    "and reply_id: '%s',"
                    "template to render: '%s'",
                    recipients,
                    sender,
                    self.get_message_id(),
                    self.get_reply_id(),
                    self.get_template_path())

        content_html, subject = self.render()

        headers = params.get('headers', {})

        msg = MessageWrapper()
        msg.set_subject(subject)
        msg.set_sender(sender)
        msg.set_to(recipients)
        msg.set_extra_headers(headers)

        if self.get_message_id():
            msg.set_message_id(self.get_message_id())
        if self.get_reply_id():
            msg.set_reply_to(self.get_reply_id())

        msg.set_body(content_html.strip().encode('utf-8'))

        msg.send_raw_email()
        return msg

    def get_message_id(self):
        return self.id

    def get_reply_id(self):
        return self.parent_id


class EmailMessage(HTMLEmailMessage):
    def render(self):
        text, subj = super(EmailMessage, self).render()
        return WikiFormatter(cfg='intranet_email').to_html(text), subj
