import re
from json import loads

from django.conf import settings

from .startrek import StartrekIssueMessage, StartrekCommentMessage
from .mail import EmailMessage, HTMLEmailMessage


class TransportMetaclass(type):

    def __new__(mcs, name, bases, attrs):
        new_class = type.__new__(mcs, name, bases, attrs)
        transport_id = new_class.transport_id
        if transport_id:
            Transport._registry[transport_id] = new_class
        return new_class


class Transport(metaclass=TransportMetaclass):
    _registry = {}
    transport_id = None

    def __init__(self, department, office, staff, route_params, thread):
        super(Transport, self).__init__()
        self.department = department
        self.office = office
        self.staff = staff
        self.thread = thread
        self.message = None
        self.params = loads(route_params) if route_params else {}

    def prepare_message(self, context, template):
        raise NotImplementedError

    def make_params(self, params):
        def merge(d1, d2):
            d1 = d1.copy()
            d2 = d2.copy()
            for k, v1 in d1.items():
                v2 = d2.pop(k, None)
                if v2 is None:
                    continue
                elif isinstance(v1, dict) and isinstance(v2, dict):
                    v1 = merge(v1, v2)
                elif isinstance(v1, list) and isinstance(v2, list):
                    v1 += v2
                else:
                    v1 = v2
                d1[k] = v1
            d1.update(d2)
            return d1

        params = merge(self.params, params)

        for k in params:
            if isinstance(k, str):
                params[str(k)] = params.pop(k)

        params = self.replace_forbidden_to_notify(params)

        return params

    @staticmethod
    def replace_forbidden_to_notify(params):
        if params.pop('notify_forbidden', None):
            return params

        to_exclude = settings.NOTIFICATIONS_FORBIDDEN_TO_NOTIFY

        def add_to_access(login):
            access = set(params.get('access', []))
            access.add(login)
            params['access'] = list(access)

        if 'assignee' in params:
            for login in to_exclude:
                if params['assignee'] == login:
                    del params['assignee']
                    add_to_access(login)
                    break

        if 'followers' in params:
            followers = set(params['followers'])
            for login in to_exclude:
                if login in followers:
                    followers.remove(login)
                    add_to_access(login)

            params['followers'] = list(followers)

        to_exlclude_re = [re.compile(r' ?{}@'.format(it)) for it in to_exclude]
        if 'recipients' in params:
            def is_top_email(email):
                return any(regex.match(email) for regex in to_exlclude_re)
            params['recipients'] = sorted(list({
                it if not is_top_email(it) else 'tops-gaps@yandex-team.ru'
                for it in params['recipients']
            }))

        return params

    def deliver_message(self, **params):
        message_params = self.make_params(params)
        self.thread.prepare_message(self.message)
        self.message.send(**message_params)
        self.thread.commit_message(self.message)


def transport_factory(transport_id):
    return Transport._registry[transport_id]


class EmailTransport(Transport):
    transport_id = 'email'

    def prepare_message(self, context, template):
        self.message = EmailMessage(context=context, template=template)


class HTMLEmailTransport(Transport):
    transport_id = 'html_email'

    def prepare_message(self, context, template):
        self.message = HTMLEmailMessage(context=context, template=template)


class FakeTransport(Transport):
    transport_id = 'fake'

    def prepare_message(self, context, template):
        pass

    def deliver_message(self, **params):
        pass


class StartrekTransport(Transport):
    def deliver_message(self, **params):
        message_params = self.make_params(params)
        self.thread.prepare_message(self.message)
        obj = self.message.send(**message_params)
        self.thread.commit_message(self.message)
        return obj


class StartrekIssueTransport(StartrekTransport):
    transport_id = 'startrek_issue'

    def prepare_message(self, context, template):
        self.message = StartrekIssueMessage(context=context,
                                            template=template)


class StartrekCommentTransport(StartrekTransport):
    transport_id = 'startrek_comment'

    def prepare_message(self, context, template):
        self.message = StartrekCommentMessage(context=context,
                                              template=template)
