from phonenumbers import PhoneNumberFormat, format_number, parse, PhoneNumberMatcher

from django import forms

from .validators import validate_phone, phone_is_valid


class PhoneMixin(object):
    default_region = 'RU'

    def __init__(self, region=None, *args, **kwargs):
        self.region = region or self.default_region
        super(PhoneMixin, self).__init__(*args, **kwargs)

    def find_numbers(self, text):
        for match in PhoneNumberMatcher(text, self.region):
            yield self.format_e164(match.number)

    def format_e164(self, parsed):
        return format_number(parsed, PhoneNumberFormat.E164)

    def format_international(self, parsed):
        return format_number(parsed, PhoneNumberFormat.INTERNATIONAL)

    def parse_and_format_e164(self, number):
        return self.format_e164(parse(number, self.region))

    def parse_and_format_international(self, number):
        return self.format_international(parse(number, self.region))


class PhoneField(PhoneMixin, forms.CharField):
    default_validators = [validate_phone]

    def clean(self, value):
        as_unicode = super(PhoneField, self).clean(value)
        if not as_unicode.strip():
            return ''
        if not phone_is_valid(as_unicode):
            return as_unicode
        return self.parse_and_format_e164(as_unicode)

    def prepare_value(self, value):
        parent_value = super(PhoneField, self).prepare_value(value)
        if not parent_value:
            return ''
        if not phone_is_valid(parent_value):
            return parent_value
        return self.parse_and_format_international(parent_value)


class MultiPhoneField(PhoneMixin, forms.CharField):

    def to_python(self, value):
        return list(self.find_numbers(value))

    def validate(self, value):
        super(MultiPhoneField, self).validate(value)
        for number in value:
            validate_phone(number)

    def prepare_value(self, value):
        value = ', '.join(value)
        return super(MultiPhoneField, self).prepare_value(value)
