from typing import Dict

from django.conf import settings
from . import RpcException, RpcResponse
from .serializers import get_serializer

_methods: Dict[str, dict] = {}


def _load_apps():
    for app in getattr(settings, 'INSTALLED_APPS', ()):
        try:
            rpc_app = '%s.rpc' % app
            __import__(rpc_app)
        except ImportError as e:
            if str(e).find('No module named') == 0:
                pass
            else:
                raise


def register_method(func, name, module=None):
    if module is None:
        # module name between project name and .rpc
        module = func.__module__.split('.')[-2]

    module_methods = _methods.setdefault(module, {})
    if name not in module_methods:
        module_methods[name] = {'func': func}


def methods_list():
    if not _methods:
        _load_apps()
    return _methods


def dispatch(type, request):
    methods = methods_list()

    response = RpcResponse()
    s = get_serializer(type)
    try:
        rpc_request = s.deserialize(request)
    except Exception as e:
        response.error = e
        return s.serialize(response, request)

    i = rpc_request.method.rfind('.')
    module, method = rpc_request.method[:i], rpc_request.method[i+1:]
    if module == 'django_rpc':
        module = 'rpc'
    if module not in methods or method not in methods[module]:
        response.error = 'Unknown method %s' % rpc_request.method
        return s.serialize(response, request)

    try:
        if 'kwargs' in methods[module][method]['func'].__code__.co_varnames:
            kwargs = {'request': request}
            response.result = methods[module][method]['func'](*rpc_request.params, **kwargs)
        else:
            response.result = methods[module][method]['func'](*rpc_request.params)
    except RpcException as e:
        response.error = e
    return s.serialize(response, request)
