import itertools
import operator

from staff.person.models import Staff

from staff.lib.serializers.xml_serializer import XMLSerializer
from staff.lib.serializers.json_serializer import JSONSerializer

from .models import EmailRedirection


class EmailRedirectionController(object):

    _base_query = EmailRedirection.objects.all()
    xml_tag_names = ['redirections', 'redirection', '', 'login']

    def _query_for_receiver(self, receiver):
        return self._base_query.filter(to_person=receiver)

    def _get_data(self):
        qs = (
            self._base_query
            .order_by('from_person__login', 'to_person__login')
            .values('from_person__login', 'to_person__login')
        )

        key_func = operator.itemgetter('from_person__login')

        return [
            {
                'from': from_login,
                'to': [redir['to_person__login'] for redir in redirections],
            }
            for from_login, redirections in itertools.groupby(qs, key_func)
        ]

    def _replace_receiver(self, just_dismissed, forward_to):
        self._query_for_receiver(just_dismissed).update(to_person=forward_to)

    def _create(self, from_person, to_person):
        self._base_query.get_or_create(from_person=from_person,
                                       to_person=to_person)

    def _drop_redirections_to(self, just_dismissed):
        self._query_for_receiver(just_dismissed).delete()

    def get_xml(self):
        data = self._get_data()
        xml = XMLSerializer(data, self.xml_tag_names).serialize()
        return xml

    def get_json(self):
        data = self._get_data()
        json = JSONSerializer(data).serialize()
        return json

    def create_redirection(self, just_dismissed, forward_correspondence_to):

        if forward_correspondence_to:
            self._replace_receiver(just_dismissed, forward_correspondence_to)
            self._create(just_dismissed, forward_correspondence_to)

        else:
            self._drop_redirections_to(just_dismissed)

    def notify_restored(self, resurrected_person):
        self._base_query.filter(from_person=resurrected_person).delete()

    def drop_redirections(self, from_person, to_person):
        (self._query_for_receiver(to_person)
             .filter(from_person=from_person).delete())

    def get_for_reciever(self, person):
        return Staff.objects.filter(emailredirections_from__to_person=person)
