from collections import defaultdict

import blackbox

from django.core.validators import validate_email
from django.core.exceptions import ValidationError

from staff.lib import requests, tvm2
from staff.person.models import Staff

from staff.emails.source_type import SOURCE_TYPE


def _entry(person__uid, email, source_type, is_main=False):
    return locals()


class Datasource:
    def __init__(self, settings, staff=None):
        super().__init__()
        self.staff = staff

        if self.staff is not None:
            self.base_staff_query = self.base_staff_query.filter(pk=self.staff.id)

    base_staff_query = Staff.objects.filter(uid__isnull=False)


class CRMDatasource(Datasource):
    def __init__(self, settings):
        super().__init__(settings)
        self.protocol = settings.CRM_PROTOCOL
        self.host = settings.CRM_HOST
        self.port = settings.CRM_PORT
        self.path = settings.CRM_PATH

    def load_emails(self):
        response = requests.get(
            url=self.get_url(),
            headers={tvm2.TVM_SERVICE_TICKET_HEADER: tvm2.get_tvm_ticket_by_deploy('crm')},
            timeout=(4, 8, 16),
        )

        for user_dict in response.json():
            if 'crm_emails' in user_dict:
                main = user_dict['main_crmemail']
                uid = str(user_dict['passport_uid'])
                for crm_email in user_dict['crm_emails']:
                    yield _entry(uid, crm_email, SOURCE_TYPE.CRM, main == crm_email)

    def get_url(self):
        url_template = '{protocol}://{host}:{port}/{path}'
        return url_template.format(**vars(self))


class PassportDatasource(Datasource):

    def load_emails(self):
        query = self.base_staff_query.values_list('uid', flat=True)

        for uid in query:
            for email in blackbox.list_emails(uid):
                yield _entry(str(uid), email, SOURCE_TYPE.PASSPORT)


class StaffDatasource(Datasource):

    def load_emails(self):
        returned_emails = defaultdict(set)

        # WORK EMAIL
        work_emails = (
            self.base_staff_query.values_list('uid', 'work_email')
        )
        for uid, email in work_emails:
            email = email.lower()

            try:
                validate_email(email)
            except ValidationError:
                continue
            returned_emails[uid].add(email)

            yield _entry(uid, email, SOURCE_TYPE.STAFF)

        # HOME EMAIL
        home_emails = (
            self.base_staff_query
            .filter(home_email__contains='@')
            .values_list('uid', 'home_email')
        )

        for uid, email in home_emails:
            email = email.lower()

            try:
                validate_email(email)
            except ValidationError:
                continue

            if email not in returned_emails[uid]:
                returned_emails[uid].add(email)
                yield _entry(uid, email, SOURCE_TYPE.STAFF)

        # BIG YANDEX EMAIL
        query = (
            self.base_staff_query
            .filter(
                is_login_passport_confirmed=True,
                login_passport__isnull=False)
            .exclude(login_passport__contains='@')
            .values_list(
                'uid',
                'login_passport',)
        )

        yandex_domains = [
            'ya.ru',
            'yandex.by',
            'yandex.com',
            'yandex.kz',
            'yandex.ru',
            'yandex.ua',
        ]

        for uid, login in query:

            for domain in yandex_domains:
                email = '{login}@{domain}'.format(login=login, domain=domain)
                email = email.lower()

                if email not in returned_emails[uid]:
                    yield _entry(uid, email, SOURCE_TYPE.STAFF)
