import logging

from django.conf import settings

from staff.celery_app import app
from staff.lib.lock import locked
from staff.person.models import Staff
from staff.person.controllers import PersonCtl
from staff.lib.sync_tools.diff_merger import DataDiffMerger
from staff.lib.sync_tools.updater import Updater
from staff.lib.tasks import LockedTask

from staff.emails.datagenerators import StaffEmailDataGenerator, PassportEmailDataGenerator, CRMEmailDataGenerator
from staff.emails.datasources import StaffDatasource, PassportDatasource, CRMDatasource
from staff.emails.models import Email

# Connect to signals, do not remove
# noinspection PyUnresolvedReferences
from staff.emission.django.emission_master.models import MasterLog  # noqa ignore=F401

logger = logging.getLogger('emails.sync_tasks')


@app.task(ignore_result=True)
@locked('sync_crm_emails')
def sync_crm_emails():
    crm_datasource = CRMDatasource(settings)
    data_generator = CRMEmailDataGenerator(crm_datasource)
    data_diff_merger = DataDiffMerger(data_generator, logger)
    updater = Updater(data_diff_merger, logger)
    updater.source_type = 'CRM email'

    updater.run_sync()
    update_persons_crm_email()


@app.task(ignore_result=True)
@locked('sync_passport_emails')
def sync_passport_emails():
    passport_datasource = PassportDatasource(settings)
    data_generator = PassportEmailDataGenerator(passport_datasource)
    data_diff_merger = DataDiffMerger(data_generator, logger)
    updater = Updater(data_diff_merger, logger)
    updater.source_type = 'Passport email'

    updater.run_sync()


@app.task
class SyncStaffEmails(LockedTask):
    def _create_updater(self, staff: Staff) -> Updater:
        staff_data_source = StaffDatasource(settings, staff)
        staff_email_data_generator = StaffEmailDataGenerator(staff_data_source)
        data_diff_merger = DataDiffMerger(staff_email_data_generator, logger)
        updater = Updater(data_diff_merger, logger)
        updater.source_type = 'Staff email'
        if staff is not None:
            updater.do_delete = False

        return updater

    def locked_run(self, staff_id=None):
        if staff_id:
            logger.info('Syncing staff emails for %s', staff_id)
        else:
            logger.info('Syncing staff emails')

        staff = Staff.objects.get(id=staff_id) if staff_id else None

        updater = self._create_updater(staff)
        updater.run_sync()


def update_persons_crm_email():
    crm_emails_qs = Email.crm.filter(is_main=True).values_list('person', 'email')

    persons = Staff.objects.filter(id__in=Email.crm.values('person'))
    persons = dict((p.id, p) for p in persons)
    persons_with_crm = set()

    for person_id, crm_email in crm_emails_qs:
        try:
            person = persons.get(person_id)
            if person:
                PersonCtl(person).update({'login_crm': crm_email}).save()

                logger.info('CRM login for person with login "%s" updated. Set "%s".', person.login, crm_email)

            persons_with_crm.add(person_id)

        except Exception:
            logger.exception('An unexpected error occurred when update CRM login for person with id %s.', person_id)

    persons = (
        Staff.objects
        .exclude(login_crm='')
        .exclude(id__in=persons_with_crm)
    )

    for person in persons:
        logger.info('CRM login removed for person with login "%s".', person.login)
        PersonCtl(person).update({'login_crm': ''}).save()
