from functools import partial

from django.db.models.signals import post_save, post_delete
from django.db import models

from staff.lib.models.base import AtomicSaveModel

from staff.emission.django.emission_master.exceptions import EmissionImproperlyConfigured
from staff.emission.django.emission_master.utils import load_model, now_str
from staff.emission.django.emission_master import settings


class MasterLog(models.Model):
    ACTION_CHOICES = [
        ('modify', 'modify'),
        ('delete', 'delete'),
    ]

    class Meta:
        verbose_name = 'Incremental log for model changes'
        verbose_name_plural = verbose_name

    data = models.TextField()
    creation_time = models.CharField(max_length=32, default=now_str, db_index=True)

    action = models.CharField(max_length=10, choices=ACTION_CHOICES, default='modify')
    batch_id = models.UUIDField(db_index=True, null=True, blank=True)

    def __str__(self):
        return '%d "%s"' % (self.id, self.data)


class NotSent(models.Model):
    entry = models.ForeignKey(MasterLog, primary_key=True, unique=True)


class Sent(models.Model):
    entry = models.ForeignKey(MasterLog, primary_key=True, unique=True)


logged_models = set()


def init_logged_models():
    if logged_models:
        return

    broken_models = []

    for model_name in settings.REPLICATED_MODELS:
        try:
            model = load_model(model_name)
            if not issubclass(model, AtomicSaveModel):
                broken_models.append(model)
            logged_models.add(model)
        except EmissionImproperlyConfigured:
            continue

    assert not broken_models, broken_models


def write_log(sender, instance, using, _action, **kwargs):
    from staff.emission.django.emission_master.controller import controller
    init_logged_models()
    if sender in logged_models:
        controller.append(obj=instance, action=_action, using=using)


post_save.connect(partial(write_log, _action='modify'), weak=False)
post_delete.connect(partial(write_log, _action='delete'), weak=False)
