import datetime
import hashlib
import re
import sys
import time

from django import http
from django.apps import apps

from staff.lib.db import atomic

from staff.emission.django.emission_master.exceptions import EmissionImproperlyConfigured


def now_str():
    return str(datetime.datetime.now())


def import_object(name):

    if callable(name):
        return name

    match = re.match(r'(.*)\.(\w+)$', name)

    if match:
        module_name, object_name = match.groups()
    else:
        msg = "'%s' doesn't match python dotted path pattern"
        raise EmissionImproperlyConfigured(msg % name)

    try:
        __import__(module_name)
        module = sys.modules[module_name]
    except ImportError:
        msg = "Module '%s' can not be imported"
        raise EmissionImproperlyConfigured(msg % module_name)

    try:
        obj = getattr(module, object_name)
    except AttributeError:
        msg = "Object '%s' not found in module '%s'"
        raise EmissionImproperlyConfigured(msg % (object_name, module_name))

    return obj


def load_model(name):
    app_name, model_name = name.rsplit('.', 1)

    model = apps.get_model(app_name, model_name)

    if model:
        return model

    return import_object(name)


def isexception(obj):
    return isinstance(obj, type) and issubclass(obj, Exception)


def get_setting(settings, name, default=EmissionImproperlyConfigured):
    try:
        return getattr(settings, name)
    except AttributeError:

        if isexception(default):
            raise default(name)

        elif callable(default):
            return default(name)

        else:
            return default


@atomic
def chunked_queryset(queryset, limit=None):
    from staff.emission.django.emission_master.settings import EMISSION_DATABASE_CHUNK_SIZE_LIMIT

    limit = limit or EMISSION_DATABASE_CHUNK_SIZE_LIMIT
    offset = 0
    while True:
        results = list(queryset[offset:offset+limit].iterator())

        if results:
            for result in results:
                yield result
            offset += limit
        else:
            break


def streaming_multipart_response(content_iterator, content_type):
    boundary = '=' * 6 + hashlib.md5(str(time.time()).encode('utf-8')).hexdigest()[:14]
    response_generator = multipart_generator(
        content_iterator=content_iterator,
        boundary=boundary,
        content_type=content_type,
    )
    response_content_type = 'multipart/mixed; boundary="{}"'.format(boundary)

    if hasattr(http, 'StreamingHttpResponse'):
        return http.StreamingHttpResponse(
            streaming_content=response_generator,
            content_type=response_content_type,
        )
    else:
        return http.HttpResponse(
            content=response_generator,
            content_type=response_content_type,
        )


def multipart_generator(content_iterator, boundary, content_type):
    PART_TEMPLATE = (
        "--{boundary}\r\n"
        "Content-Type: {content_type}\r\n"
        "\r\n"
        "{content}\r\n"
        "\r\n"
    )

    for content in content_iterator:
        yield PART_TEMPLATE.format(
            boundary=boundary,
            content_type=content_type,
            content=content,
        )

    yield "--{}--\r\n".format(boundary)
