import json
import logging

from typing import Collection

try:
    import uwsgi
except ImportError:
    uwsgi = None

from django import http
from django.apps import apps
from django.conf import settings
from django.db.models.query import QuerySet
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_POST

try:
    from mptt.models import MPTTModel
except ImportError:
    MPTTModel = None

from staff.lib.decorators import available_by_tvm

from staff.emission.django.emission_master.utils import now_str
from staff.emission.django.emission_master import forms
from staff.emission.django.emission_master.controller import controller
from staff.emission.django.emission_master.utils import streaming_multipart_response, chunked_queryset


logger = logging.getLogger(__name__)


CONTENT_TYPE_JSON = "application/json; charset=utf-8"


@available_by_tvm(['staff-api'])
def log_slice(request):
    form = forms.SliceForm(request.GET)

    if form.is_valid():
        start = form.cleaned_data['start']
        stop = form.cleaned_data['stop']

        try:
            data = controller[start:stop]
        except KeyError:
            data = []

        if form.cleaned_data['multipart']:
            return streaming_multipart_response(
                content_iterator=(json.dumps(row) for row in data),
                content_type=CONTENT_TYPE_JSON,
            )
        else:
            return http.HttpResponse(
                content=json.dumps(list(data)),
                content_type=CONTENT_TYPE_JSON
            )

    return http.HttpResponseBadRequest(form.errors.as_text())


@available_by_tvm(['staff-api'])
def meta(request):
    form = forms.MetaForm(request.GET)

    if not form.is_valid():
        return http.HttpResponseBadRequest(form.errors.as_text())

    data = {
        'last_message': controller.get_one(controller.get_last_id()),
        'entities': forms.MODELS,
    }

    current_id = form.cleaned_data['current']
    if current_id:
        next_id = controller.get_next_id(current_id)
        if next_id:
            data['next_message'] = controller.get_one(next_id)

    return http.HttpResponse(json.dumps(data),
                             content_type=CONTENT_TYPE_JSON)


def message_generator(queryset):
    for obj in queryset:
        message = {
            'id': 0,
            'action': 'modify',
            'creation_time': now_str(),
            'data': controller.storage.serialize_objects([obj]),
        }
        yield json.dumps(message)


@available_by_tvm(['staff-api'])
def entities_data_paginated(request):
    form = forms.EntityForm(request.GET)

    if not form.is_valid():
        return http.HttpResponseBadRequest(form.errors.as_text())

    if uwsgi:
        harakiri_timeout = getattr(settings, 'EMISSION_HARAKIRI_TIMEOUT', 600)
        uwsgi.set_user_harakiri(harakiri_timeout)

    continuation_token = form.cleaned_data['continuation_token']
    app_label, model_name = form.cleaned_data['name'].split('.', 1)
    model = apps.get_model(app_label, model_name)

    qs = model._default_manager.all()
    result_continuation_token = None
    if isinstance(qs, QuerySet):
        if MPTTModel and issubclass(model, MPTTModel):
            result = qs.order_by('tree_id', 'lft')
            result_continuation_token = None
        else:
            if continuation_token:
                qs = qs.filter(pk__gt=continuation_token)
            chunk_size = getattr(settings, 'EMISSION_ENTITIES_CHUNK_SIZE', 20000)
            result = list(qs.order_by('pk')[:chunk_size])

            result_continuation_token = result[-1].pk if result else None

    message = {
        'id': 0,
        'action': 'modify',
        'creation_time': now_str(),
        'data': controller.storage.serialize_objects(result),
        'continuation_token': result_continuation_token,
    }

    return http.JsonResponse(message)


@available_by_tvm(['staff-api'])
def entities_data(request):
    form = forms.EntityForm(request.GET)

    if form.is_valid():
        if uwsgi:
            harakiri_timeout = getattr(settings, 'EMISSION_HARAKIRI_TIMEOUT', 600)
            uwsgi.set_user_harakiri(harakiri_timeout)

        app_label, model_name = form.cleaned_data['name'].split('.', 1)

        model = apps.get_model(app_label, model_name)

        qs = model._default_manager.all()
        if isinstance(qs, QuerySet):
            if MPTTModel and issubclass(model, MPTTModel):
                qs = qs.order_by('tree_id', 'lft')
            else:
                qs = qs.order_by('pk')
            qs = chunked_queryset(qs)

        if form.cleaned_data['multipart']:
            return streaming_multipart_response(
                content_iterator=message_generator(qs),
                content_type=CONTENT_TYPE_JSON,
            )
        else:
            message = {
                'id': 0,
                'action': 'modify',
                'creation_time': now_str(),
                'data': controller.storage.serialize_objects(qs)
            }

            return http.HttpResponse(
                content=json.dumps(message),
                content_type=CONTENT_TYPE_JSON,
            )

    return http.HttpResponseBadRequest(form.errors.as_text())


@available_by_tvm(['staff-api'])
def get_unsent(request):
    form = forms.GetUnsentForm(request.GET)

    if not form.is_valid():
        return http.HttpResponseBadRequest(form.errors.as_text())

    count = form.cleaned_data['count'] if form.cleaned_data['count'] else settings.EMISSION_DATABASE_CHUNK_SIZE_LIMIT
    data = controller.get_unsent(count)
    return http.JsonResponse(data=list(data), safe=False)


@require_POST
@csrf_exempt
@available_by_tvm(['staff-api'])
def mark_sent(request):
    message_ids = request.POST.getlist('message_ids')

    if not message_ids:
        return http.HttpResponseBadRequest()

    if not isinstance(message_ids, Collection):
        return http.HttpResponseBadRequest()

    for value in message_ids:
        if not value.isdigit():
            return http.HttpResponseBadRequest()

        if int(value) <= 0:
            return http.HttpResponseBadRequest()

    logger.info('Marking sent %s messages', len(message_ids))
    controller.mark_sent(message_ids)
    return http.HttpResponse(status=204)
