import logging

from dateutil.parser import parse

from django.conf import settings

from staff.femida.femida_session import femida_session

logger = logging.getLogger(__name__)


class VacanciesDatasource(object):
    def __init__(self, **kwargs):
        pass

    def __iter__(self):
        next_page = f'https://{settings.FEMIDA_HOST}/_api/staff/vacancies/?page_size=1000'

        while next_page:
            data = self._request(next_page)
            next_page = data['next']

            for obj in data['results']:
                yield self._process_vacancy(obj)

    def _request(self, url):
        response = femida_session.get(url, timeout=(1, 3, 5))

        if not response.status_code == 200:
            raise RuntimeError("Femida response not 200")

        return response.json()

    def _process_vacancy(self, vacancy):
        vacancy_id = vacancy['id']

        try:
            result = {
                'id': vacancy_id,
                'name': vacancy['name'],
                'status': vacancy['status'],
                'created': parse(vacancy['created'], ignoretz=True),
                'modified': parse(vacancy['modified'], ignoretz=True),
                'startrek_key': vacancy.get('startrek_key'),
                'headcount_position_id': vacancy.get('budget_position_id'),
                'is_published': vacancy['is_published'],
                'is_hidden': vacancy['is_hidden'],
                'profession_staff_id': vacancy['profession_staff_id'],
                'department_url': vacancy['department_url'],
                'members': vacancy['access'],
            }

            if not vacancy.get('offer'):
                return result

            offer_data = vacancy['offer']
            result['offer_id'] = offer_data['id']
            result['preprofile_id'] = offer_data.get('newhire_id')
            result['candidate_login'] = offer_data.get('username', '')
            result['application_id'] = offer_data.get('application_id', None)
            result['startrek_salary_key'] = offer_data.get('startrek_salary_key')

            if not offer_data.get('candidate'):
                return result

            candidate_data = offer_data['candidate']
            result['candidate_id'] = candidate_data['id']
            # First name is not required field in femida
            result['candidate_first_name'] = candidate_data['first_name'] or ''
            result['candidate_last_name'] = candidate_data['last_name'] or ''
        except Exception:
            logger.error('Error while parsing vacancy with id %s', vacancy_id)
            raise

        return result
