from typing import Dict

from staff.lib import waffle
from collections import defaultdict

from django.db.models import Q

from staff.lib.db import atomic
from staff.lib.sync_tools.rollupper import Rollupper
from staff.lib.sync_tools.data_mapper import DataMapper

from staff.departments.models import Department, Vacancy, VacancyMember, DepartmentStaff, DepartmentRoles
from staff.preprofile.models import Preprofile
from staff.budget_position.models import BudgetPosition
from staff.person.models import Staff, Occupation

from staff.femida.models import FemidaVacancy
from staff.femida.constants import VACANCY_STATUS


class MapperCache:
    preprofile_cache: Dict = {}
    budget_position_cache: Dict = {}
    department_cache: Dict = {}
    occupation_cache: Dict = {}

    @classmethod
    def reset_cache(cls):
        cls.preprofile_cache = dict(Preprofile.objects.values_list('id', 'id'))
        cls.budget_position_cache = dict(BudgetPosition.objects.values_list('code', 'id'))
        cls.department_cache = dict(Department.objects.values_list('url', 'id'))
        cls.occupation_cache = dict(Occupation.objects.values_list('name', 'name'))


def status_converter(status):
    active_statuses = {
        VACANCY_STATUS.IN_PROGRESS,
        VACANCY_STATUS.OFFER_PROCESSING,
        VACANCY_STATUS.ON_APPROVAL,
        VACANCY_STATUS.SUSPENDED,
        VACANCY_STATUS.OFFER_ACCEPTED,
    }

    return status in active_statuses


class VacanciesMapper(DataMapper):
    def __init__(self, object_dict: Dict):
        super().__init__(object_dict)

        self.mapping = (
            ('name', 'name'),
            ('startrek_key', 'ticket'),
            ('startrek_salary_key', 'salary_ticket'),
            ('offer_id', 'offer_id'),
            ('candidate_id', 'candidate_id'),
            ('candidate_first_name', 'candidate_first_name'),
            ('candidate_last_name', 'candidate_last_name'),
            ('candidate_login', 'candidate_login'),
            ('headcount_position_id', 'headcount_position_code'),
            ('headcount_position_id', 'budget_position_id', self.budget_position_converter),
            ('status', 'status'),
            ('status', 'is_active', status_converter),
            ('application_id', 'application_id'),
            ('is_published', 'is_published'),
            ('is_hidden', 'is_hidden'),
            ('profession_staff_id', 'occupation_id', self.occupation_converter),
            ('preprofile_id', 'preprofile_id', self.preprofile_converter),
            ('department_url', 'department_id', self.department_converter),
        )

    def preprofile_converter(self, preprofile_id):
        return MapperCache.preprofile_cache.get(preprofile_id, None)

    def department_converter(self, department_url):
        return MapperCache.department_cache.get(department_url, None)

    def occupation_converter(self, profession_staff_id):
        return MapperCache.occupation_cache.get(profession_staff_id, None)

    def budget_position_converter(self, bp_code):
        if bp_code is not None and bp_code not in MapperCache.budget_position_cache:
            budget_position, _ = BudgetPosition.objects.get_or_create(code=bp_code)
            MapperCache.budget_position_cache[budget_position.code] = budget_position.id
        return MapperCache.budget_position_cache.get(bp_code, None)


class VacanciesRolluper(Rollupper):
    data_mapper_class = VacanciesMapper
    model = FemidaVacancy
    link_field_name = 'staff_vacancy'
    key_field_name = 'id'

    def __init__(self, logger=None, debug=False, create_absent=False):
        super().__init__(logger, debug, create_absent)
        MapperCache.reset_cache()

    def run_rollup(self, object_id=None, dry_run=False):
        super().run_rollup(object_id, dry_run)
        if dry_run:
            return

        filter_ = Q()
        if not waffle.switch_is_active('enable_proposal_vacancies_for_everyone'):
            filter_ &= Q(login__in=(
                DepartmentStaff.objects
                .filter(role_id=DepartmentRoles.HR_ANALYST.value)
                .values_list('staff__login', flat=True)
            ))
        cache = dict(Staff.objects.filter(filter_).values_list('login', 'id'))
        self.logger.logger.info('Syncing vacancy members')

        with atomic():
            members = defaultdict(dict)
            for member in VacancyMember.objects.values('id', 'vacancy_id', 'person_id'):
                members[member['vacancy_id']][member['person_id']] = member['id']

            to_create = []
            to_delete = []

            for vacancy in FemidaVacancy.objects.all():
                current_members = set(members.get(vacancy.id, {}).keys())
                should_be = {cache[member] for member in vacancy.members if member in cache}
                to_delete.extend([
                    members[vacancy.id][person_id]
                    for person_id in current_members - should_be
                ])
                to_create.extend([
                    VacancyMember(vacancy_id=vacancy.id, person_id=person_id)
                    for person_id in should_be - current_members
                ])

            VacancyMember.objects.filter(id__in=to_delete).delete()
            self.logger.logger.info('Deleted %d vacancy members', len(to_delete))
            VacancyMember.objects.bulk_create(to_create)
            self.logger.logger.info('Created %d vacancy members', len(to_create))

    def create_dis_instance(self, oebs_instance):
        return Vacancy(
            id=oebs_instance.id,
            name=oebs_instance.name,
            is_published=False,
        )
