import csv
from datetime import timedelta, datetime
from typing import Optional

import sform

from django.conf import settings
from django.core.exceptions import ValidationError
from django.forms.fields import MultipleChoiceField as DjMultipleChoiceField
from django import forms

from staff.lib.utils.ordered_choices import OrderedChoices
from staff.lib.utils.enum_choices import EnumChoices
from staff.person.models.person import Staff

from staff.gap.api.exceptions import (
    NonUniqueLoginsError,
    NoLoginColumnError,
    NoDateFromColumn,
    WrongDateFormat,
    NonexistentLogin,
    AlreadyHasMandatoryVacationError,
)
from staff.gap.controllers.gap import GapCtl
from staff.gap.controllers.utils import replace_useless_smt
from staff.gap.workflows.choices import GAP_STATES as GS

WORKFLOWS_TO_CREATE = OrderedChoices(
    ('ABSENCE', 'absence'),
    ('REMOTE_WORK', 'remote_work'),
    ('OFFICE_WORK', 'office_work'),
    ('ILLNESS', 'illness'),
    ('LEARNING', 'learning'),
    ('MATERNITY', 'maternity'),
    ('VACATION', 'vacation'),
    ('PAID_DAY_OFF', 'paid_day_off'),
    ('DUTY', 'duty'),
)

DATETIME_INPUT_FORMATS = (
    '%Y-%m-%dT%H:%M:%S.%fZ',  # '2006-10-25T14:30:59.000Z'
    '%Y-%m-%dT%H:%M:%S',      # '2006-10-25T14:30:59'
    '%Y-%m-%dT%H:%M',         # '2006-10-25T14:30'
    '%Y-%m-%d',               # '2006-10-25'
)


class PeriodicType(EnumChoices):
    WEEK = 'week'  # повтор через n недель
    # DAY = 'day'  # повтор через n дней (будет сделано в STAFF-15423)


class WEEKDAYS(EnumChoices):
    Monday = '0'
    Tuesday = '1'
    Wednesday = '2'
    Thursday = '3'
    Friday = '4'
    Saturday = '5'
    Sunday = '6'


class NVMultipleChoiceField(DjMultipleChoiceField):

    def validate(self, value):
        pass


class SFMultipleChoiceField(sform.MultipleChoiceField):

    def get_dj_field(self, required=False):
        return NVMultipleChoiceField(
            required=required,
            choices=self.choices,
            **self.kwargs
        )


class BaseGapForm(sform.SForm):
    full_day = sform.BooleanField(
        state=sform.REQUIRED,
        default=False,
    )
    date_from = sform.DateTimeField(
        state=sform.REQUIRED,
        input_formats=DATETIME_INPUT_FORMATS,
    )
    date_to = sform.DateTimeField(
        state=sform.REQUIRED,
        input_formats=DATETIME_INPUT_FORMATS,
    )
    work_in_absence = sform.BooleanField(
        state=sform.REQUIRED,
        default=False,
    )
    comment = sform.CharField(
        state=sform.NORMAL,
        default='',
    )
    to_notify = SFMultipleChoiceField(
        state=sform.NORMAL,
        choices=[],
        default=[],
    )

    def clean_to_notify(self, value):
        return value if value else []

    def clean_date_from(self, value):
        return replace_useless_smt(value)

    def clean_date_to(self, value):
        date_from = self.cleaned_data['date_from']
        date_to = value
        full_day = self.cleaned_data['full_day']

        if date_to < date_from:
            raise ValidationError(
                'date_to is less than date_from',
                code='invalid',
            )

        if date_to == date_from and not full_day:
            raise ValidationError(
                'date_to is equal to date_from and it is not full_day',
                code='invalid',
            )

        return replace_useless_smt(date_to)


class BasePeriodicGapForm(BaseGapForm):
    """
    Форма для создания периодических отсутствий
    В случае создания одиночного события будут игнорироваться поля этой формы
    Пример:
    period=2, type=WEEK, periodic_map_weekdays=[0,3], periodic_date_to=2022-01-01
     - это значит повторять каждую вторую неделю по понедельникам и четвергам до 2022-01-01
    """
    is_periodic_gap = sform.BooleanField(
        state=sform.REQUIRED,
        default=False,
    )
    periodic_type = sform.ChoiceField(
        choices=PeriodicType.choices(),
        default=PeriodicType.WEEK.value,
    )
    period = sform.IntegerField(state=sform.NORMAL, default=1)
    periodic_map_weekdays = sform.MultipleChoiceField(
        choices=WEEKDAYS.choices(),
        state=sform.NORMAL,
    )
    periodic_date_to = sform.DateTimeField(
        state=sform.NORMAL,
        input_formats=DATETIME_INPUT_FORMATS,
    )

    def clean_periodic_date_to(self, value):
        is_periodic_gap = self.cleaned_data['is_periodic_gap']
        if not is_periodic_gap:
            return None
        if value is None:
            raise ValidationError(
                'periodic_date_to is required',
                code='periodic_date_to_empty',
            )
        date_to = self.cleaned_data['date_to']
        periodic_date_to = value

        if date_to > periodic_date_to:
            raise ValidationError(
                'periodic_date_to is less than date_to',
                code='invalid',
            )

        if periodic_date_to > date_to + timedelta(days=31 * 3):
            raise ValidationError(
                'periodic_date_to is more than date_to + 3 month',
                code='invalid',
            )

        return replace_useless_smt(periodic_date_to)

    def clean_periodic_map_weekdays(self, value):
        is_periodic_gap = self.cleaned_data['is_periodic_gap']
        if not is_periodic_gap:
            return None
        if not value:
            raise ValidationError(
                'periodic_map_weekdays is required in periodic gap',
                code='invalid',
            )
        return sorted(value)


'''
Формы отсутствий
'''


class AbsenceForm(sform.SForm):
    pass


class RemoteWorkForm(sform.SForm):
    pass


class OfficeWorkForm(sform.SForm):
    pass


class IllnessForm(sform.SForm):
    has_sicklist = sform.BooleanField(
        state=sform.REQUIRED,
        default=False,
    )
    is_covid = sform.BooleanField(
        state=sform.REQUIRED,
        default=False,
    )


class LearningForm(sform.SForm):
    pass


class MaternityForm(sform.SForm):
    pass


class VacationForm(sform.SForm):
    is_selfpaid = sform.BooleanField(
        state=sform.REQUIRED,
        default=False,
    )


class PaidDayOffForm(sform.SForm):
    pass


class DutyForm(sform.SForm):
    service_slug = sform.CharField(state=sform.NORMAL, default='', max_length=512)
    service_name = sform.CharField(state=sform.NORMAL, default='', max_length=512)
    shift_id = sform.IntegerField(state=sform.NORMAL, default=0)
    role_on_duty = sform.CharField(state=sform.NORMAL, default='', max_length=512)
    person_login = sform.CharField(state=sform.NORMAL, default='', max_length=128)
    person_id = sform.IntegerField(state=sform.NORMAL, default=0)


class GapCreateForm(BasePeriodicGapForm, AbsenceForm, RemoteWorkForm,
                    OfficeWorkForm, IllnessForm, LearningForm, MaternityForm,
                    VacationForm, PaidDayOffForm, DutyForm):
    person_login = sform.CharField(
        state=sform.REQUIRED,
        min_length=2,
        max_length=100,
    )
    workflow = sform.ChoiceField(
        state=sform.REQUIRED,
        choices=WORKFLOWS_TO_CREATE,
        default=WORKFLOWS_TO_CREATE.ABSENCE,
    )


'''
Формы для редактирования отсутствий
'''


class GapEditForm(BaseGapForm, AbsenceForm, RemoteWorkForm,
                  OfficeWorkForm, IllnessForm, LearningForm,
                  MaternityForm, VacationForm, DutyForm):
    gap_id = sform.IntegerField(
        state=sform.REQUIRED,
    )
    periodic_gap_id = sform.IntegerField(
        state=sform.NORMAL,
    )


'''
Формы для поиска отсутствий
'''


class GapsFindPostForm(sform.SForm):
    person_logins = SFMultipleChoiceField(
        state=sform.NORMAL,
        choices=[],
        default=[],
    )
    states = SFMultipleChoiceField(
        state=sform.NORMAL,
        choices=[],
        default=[],
    )
    workflows = SFMultipleChoiceField(
        state=sform.NORMAL,
        choices=[],
        default=None,
    )
    date_from = sform.DateTimeField(
        state=sform.NORMAL,
        input_formats=DATETIME_INPUT_FORMATS,
    )
    date_to = sform.DateTimeField(
        state=sform.NORMAL,
        input_formats=DATETIME_INPUT_FORMATS,
    )
    limit = sform.IntegerField(
        state=sform.NORMAL,
        default=100,
    )
    page = sform.IntegerField(
        state=sform.NORMAL,
        default=0,
    )


class GapsFindGetForm(sform.SForm):
    l = SFMultipleChoiceField(  # noqa ignore=E741
        state=sform.NORMAL,
        choices=[],
        default=[],
    )
    s = SFMultipleChoiceField(
        state=sform.NORMAL,
        choices=[],
        default=[],
    )
    w = SFMultipleChoiceField(
        state=sform.NORMAL,
        choices=[],
        default=None,
    )
    date_from = sform.DateTimeField(
        state=sform.NORMAL,
        input_formats=DATETIME_INPUT_FORMATS,
    )
    date_to = sform.DateTimeField(
        state=sform.NORMAL,
        input_formats=DATETIME_INPUT_FORMATS,
    )
    limit = sform.IntegerField(
        state=sform.NORMAL,
        default=100,
    )
    page = sform.IntegerField(
        state=sform.NORMAL,
        default=0,
    )


class ImportMandatoryVacationsForm(forms.Form):
    import_file = forms.FileField(max_length=settings.MANDATORY_VACATIONS_IMPORT_FILE_MAX_SIZE)

    def clean(self):
        file = self.cleaned_data.get('import_file')
        persons_data = self.parse_vacations_csv(file)

        if self.errors:
            return

        self._extend_persons_data(persons_data)
        self._check_persons_dont_have_mandatory_vacations(persons_data)
        self.cleaned_data['mandatory_vacations_data'] = persons_data

    def _get_login_from_row(self, row) -> Optional[str]:
        try:
            return row.pop('login')
        except KeyError:
            self.add_error('import_file', NoLoginColumnError())
        return None

    def parse_vacations_csv(self, file) -> dict:
        reader = csv.DictReader(
            file.read().decode('utf-8-sig').splitlines(),
            delimiter=str(';'),
        )
        result = {}

        for row in reader:
            login = self._get_login_from_row(row)

            if login is None:
                continue

            if login in result:
                self.add_error('import_file', NonUniqueLoginsError(login))

            if 'date_from' not in row:
                self.add_error('import_file', NoDateFromColumn(login))

            try:
                if 'date_from' in row:
                    datetime.strptime(
                        row['date_from'],
                        settings.MANDATORY_VACATIONS_IMPORT_FILE_DATE_FORMAT,
                    )
                if row.get('date_to'):
                    datetime.strptime(
                        row['date_to'],
                        settings.MANDATORY_VACATIONS_IMPORT_FILE_DATE_FORMAT,
                    )
            except ValueError:
                self.add_error('import_file', WrongDateFormat(login))

            result[login] = row

        return result

    def _extend_persons_data(self, persons_data):
        staff_data = (
            Staff.objects
            .filter(login__in=persons_data.keys())
            .values_list('id', 'login', 'office__city__geo_id')
        )

        diff = set(persons_data.keys()) - set(login for (_, login, _) in staff_data)
        if diff:
            for login in diff:
                self.add_error('import_file', NonexistentLogin(login))
        else:
            for person_id, login, geo_id in staff_data:
                persons_data[login]['id'] = person_id
                persons_data[login]['geo_id'] = geo_id

    def _check_persons_dont_have_mandatory_vacations(self, persons_data):
        mandatory_vacations_for_persons = GapCtl().find(
            {
                'workflow': 'vacation',
                'mandatory': True,
                'state': {'$in': [GS.NEW, GS.CONFIRMED, GS.SIGNED]},
                'person_login': {'$in': list(persons_data.keys())},
            },
            fields=['id', 'person_login', 'date_from', 'date_to'],
        )

        for vacation in mandatory_vacations_for_persons:
            login = vacation['person_login']
            new_vacation_year = datetime.strptime(
                persons_data[login]['date_from'],
                settings.MANDATORY_VACATIONS_IMPORT_FILE_DATE_FORMAT,
            ).year

            if vacation['date_from'].year == new_vacation_year:
                self.add_error('import_file', AlreadyHasMandatoryVacationError(login, vacation['id']))

    def errors_as_dict(self) -> dict:
        data = {'errors': []}
        errors = self.errors.as_data()['import_file']
        for error in errors:
            data['errors'].append(
                {
                    'code': error.code,
                    'params': error.params,
                },
            )
        return data
