from datetime import datetime, timedelta
import json
import pymongo
from pytz import timezone
from typing import List, Dict

from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_http_methods
from django.http import HttpResponseBadRequest

from staff.person.models import Staff
from staff.person_profile.controllers.calendar import _ask_calendar, CalendarError  # TODO: поменять после STAFF-11048

from staff.lib.decorators import responding_json, available_for_external
from staff.lib.utils.date import parse_datetime
from staff.lib.calendar import get_holidays

from staff.gap.controllers.gap import GapCtl, GapQueryBuilder
from staff.gap.controllers.utils import datetime_from_utc, datetime_to_utc
from staff.gap.workflows.choices import LIMITED_STATES
from staff.gap.api.availability import Availability


import logging
logger = logging.getLogger('staff.gap.api.views.availability_views')


@csrf_exempt
@require_http_methods(['GET', 'POST'])
@responding_json
@available_for_external('gap.robot_with_gap_api_access')
def availability(request):
    try:
        if request.method == 'GET':
            params: Dict[str, any] = _gaps_params_get(request.GET)
        else:
            params: Dict[str, any] = _gaps_params_post(request.body)

        params.update({
            'observer_uid': request.user.get_profile().uid,
            'observer_tvm_ticket': request.yauser.raw_user_ticket,
        })
    except (TypeError, ValueError):
        return HttpResponseBadRequest()

    meta_data = {
        'now': params['now'],
        'date_from': params['date_from'],
        'date_to': params['date_to'],
        'include_holidays': params['include_holidays'],
        'include_calendar': params['include_calendar'],
        'working_hour_from': params['working_hour_from'],
        'working_hour_to': params['working_hour_to'],
    }

    availabilities = get_availabilities(**params)

    return {
        'meta': meta_data,
        'persons': {
            person_login: _fill_availability(availability, params['now'],)
            for person_login, availability in availabilities.items()
        },
    }


def get_availabilities(
        person_logins: List[str],
        date_from: datetime,
        date_to: datetime,
        include_holidays: bool,
        include_calendar: bool,
        working_hour_from: int,
        working_hour_to: int,
        ignore_work_in_absence: bool,
        **kwargs
) -> Dict[str, Availability]:

    persons = _get_persons(person_logins)
    person_logins = list(persons.keys())
    person_geo_ids = {login: data['geo_id'] for login, data in persons.items()}
    person_timezones = {login: timezone(data['tz']) for login, data in persons.items()}

    availabilities = {
        person_login: Availability(
            date_from=date_from,
            date_to=date_to,
            timezone=person_timezones[person_login],
            working_hour_from=working_hour_from,
            working_hour_to=working_hour_to,
            ignore_work_in_absence=ignore_work_in_absence,
        ) for person_login in person_logins
    }

    if include_holidays:
        holidays = get_holidays(date_from, date_to, set(person_geo_ids.values()), out_mode='holidays')
        for login, _availability in availabilities.items():
            _availability.apply_holidays(
                holidays.get(person_geo_ids[login] if login in person_geo_ids else 225)  # Россия
            )

    for gap in _get_gaps(person_logins, date_from, date_to):
        person_login = gap['person_login']
        availabilities[person_login].add_gap(gap)

    for _availability in availabilities.values():
        _availability.apply_gaps()

    if include_calendar:
        login_by_email = {data['email']: login for login, data in persons.items()}
        for email, events in _get_calendar_events(
            observer_uid=kwargs.get('observer_uid'),
            observer_tvm_ticket=kwargs.get('observer_tvm_ticket'),
            person_emails=list(login_by_email.keys()),
            date_from=date_from,
            date_to=date_to,
        ).items():
            login = login_by_email[email]
            availabilities[login].apply_calendar_events(events)

    return availabilities


def _get_gaps(person_logins, date_from, date_to):
    gqb = (
        GapQueryBuilder()
        .person_logins(person_logins)
        .dates_not_strict(date_from, date_to)
        .workflows_states(LIMITED_STATES)
    )

    fields = [
        'id',
        'person_login',
        'workflow',
        'date_from',
        'date_to',
        'full_day',
        'work_in_absence',
    ]

    sorting = [('date_from', pymongo.ASCENDING)]

    return GapCtl().find_gaps(query=gqb.query(), fields=fields, sorting=sorting)


def _get_calendar_events(
        observer_uid: str,
        observer_tvm_ticket: str,
        person_emails: List[str],
        date_from: datetime,
        date_to: datetime) -> Dict[str, List]:
    params = {
        'from': date_from.strftime('%Y-%m-%dT%H:%M:%S%z'),
        'to': date_to.strftime('%Y-%m-%dT%H:%M:%S%z'),
        'uid': observer_uid,
        'emails': person_emails,
        'display': 'events',
        'shape': 'ids-only',
        'tz': 'UTC',
    }

    result = {}  # type: Dict[str, List]

    try:
        response = _ask_calendar(
            handle='get-availability-intervals',
            params=params,
            observer_tvm_ticket=observer_tvm_ticket,
        )
    except CalendarError:
        return result

    if response.status_code == 200:
        events = response.json()['subjectAvailabilities']
        for person_events in events:
            result[person_events['email']] = person_events.get('intervals', [])

    return result


def _gaps_params_get(params) -> Dict[str, any]:
    person_logins = params.getlist('l', [])
    include_holidays = bool(int(params.get('use_calendar', 1)))  # для обратной совместимости
    include_holidays = bool(int(params.get('include_holidays', include_holidays)))
    include_calendar = bool(int(params.get('include_calendar', 0)))
    ignore_work_in_absence = bool(int(params.get('ignore_work_in_absence', 0)))
    now = parse_datetime(params.get('now')) or datetime.utcnow()
    date_from = parse_datetime(params.get('date_from')) or now
    date_to = parse_datetime(params.get('date_to')) or date_from + timedelta(days=1)
    working_hour_from = int(params.get('working_hour_from')) if 'working_hour_from' in params else None
    working_hour_to = int(params.get('working_hour_to')) if 'working_hour_to' in params else None

    now = datetime_to_utc(now)
    date_from = datetime_to_utc(date_from)
    date_to = datetime_to_utc(date_to)

    return {
        'person_logins': person_logins,
        'include_holidays': include_holidays,
        'include_calendar': include_calendar,
        'ignore_work_in_absence': ignore_work_in_absence,
        'now': now,
        'date_from': date_from,
        'date_to': date_to,
        'working_hour_from': working_hour_from,
        'working_hour_to': working_hour_to,
    }


def _gaps_params_post(body) -> Dict[str, any]:
    data = json.loads(body)

    person_logins = data.get('person_logins', [])
    include_holidays = data.get('use_calendar', True)  # для обратной совместимости
    include_holidays = data.get('include_holidays', include_holidays)
    include_calendar = data.get('include_calendar', False)
    ignore_work_in_absence = data.get('ignore_work_in_absence', False)
    now = parse_datetime(data.get('now')) or datetime.utcnow()
    date_from = parse_datetime(data.get('date_from')) or now
    date_to = parse_datetime(data.get('date_to')) or date_from + timedelta(days=1)
    working_hour_from = int(data.get('working_hour_from')) if 'working_hour_from' in data else None
    working_hour_to = int(data.get('working_hour_to')) if 'working_hour_to' in data else None

    now = datetime_to_utc(now)
    date_from = datetime_to_utc(date_from)
    date_to = datetime_to_utc(date_to)

    return {
        'person_logins': person_logins,
        'include_holidays': include_holidays,
        'include_calendar': include_calendar,
        'ignore_work_in_absence': ignore_work_in_absence,
        'now': now,
        'date_from': date_from,
        'date_to': date_to,
        'working_hour_from': working_hour_from,
        'working_hour_to': working_hour_to,
    }


def _fill_availability(_availability, now):
    now = datetime_from_utc(now, _availability.timezone)
    available_now = _availability.is_available(now)

    data = {
        'available_now': available_now,
        'availability': [{'date_from': link.date_from, 'date_to': link.date_to} for link in _availability],
    }

    total_seconds = _availability.total_seconds()
    available_seconds = _availability.available_seconds()

    data['total_seconds'] = total_seconds
    data['available_seconds'] = available_seconds
    data['available_to_total'] = available_seconds // (total_seconds // 100)
    data['tz'] = str(_availability.timezone)

    return data


def _get_persons(logins):
    persons = (
        Staff.objects
        .filter(login__in=logins)
        .values(
            'login',
            'work_email',
            'office__city__geo_id',
            'tz',
        )
    )
    return {
        p['login']: {
            'email': p['work_email'],
            'geo_id': p['office__city__geo_id'],
            'tz': p['tz'],
        } for p in persons
    }
