import dateutil.parser

from django.conf import settings
from django_yauth.authentication_mechanisms.tvm import TvmServiceRequest
from django.contrib.auth.decorators import login_required
from django.http import (
    HttpResponseServerError,
    HttpResponseNotFound,
    HttpResponseForbidden,
    HttpResponseBadRequest,
)
from django.views.decorators.http import require_GET

from staff.lib.decorators import responding_json, available_for_external, available_by_tvm

from staff.gap.controllers.gap import GapCtl, GapQueryBuilder
from staff.gap.controllers.utils import get_short_person_with_department
from staff.gap.exceptions import GapError
from staff.gap.permissions import can_see_gap
from staff.gap.workflows.choices import LIMITED_STATES
from staff.gap.workflows.decorators import valid_gap_id


import logging
logger = logging.getLogger('staff.gap.api.views.gap_find_views')


BASE_WORKFLOW_FIELDS = (
    'id',
    'workflow',
    'person_login',
    'date_from',
    'date_to',
    'full_day',
    'work_in_absence',
    'comment',
    'to_notify',
    'state',
)

WORKFLOW_FIELDS = {
    'absence': BASE_WORKFLOW_FIELDS,
    'remote_work': BASE_WORKFLOW_FIELDS,
    'office_work': BASE_WORKFLOW_FIELDS + (
        'place',
        'office',
        'room',
        'coworking',
        'table',
    ),
    'illness': BASE_WORKFLOW_FIELDS + (
        'has_sicklist',
        'is_covid',
    ),
    'learning': BASE_WORKFLOW_FIELDS,
    'maternity': BASE_WORKFLOW_FIELDS,
    'vacation': BASE_WORKFLOW_FIELDS + (
        'is_selfpaid',
        'mandatory',
        'vacation_updated',
        'deadline',
        'geo_id',
    ),
    'trip': BASE_WORKFLOW_FIELDS,
    'conference_trip': BASE_WORKFLOW_FIELDS,
    'paid_day_off': BASE_WORKFLOW_FIELDS,
    'conference': BASE_WORKFLOW_FIELDS,
    'duty': BASE_WORKFLOW_FIELDS + (
        'service_slug',
        'service_name',
        'shift_id',
        'role_on_duty',
    ),
}


@responding_json
@login_required
@require_GET
@valid_gap_id
@available_for_external('gap.robot_with_gap_api_access')
def gap_find(request, gap_id):
    try:
        gap_id = int(gap_id)
    except (TypeError, ValueError):
        return HttpResponseBadRequest

    try:
        gap = GapCtl().find_gap_by_id(gap_id)
    except GapError:
        return HttpResponseNotFound()

    observer = request.user

    person = get_short_person_with_department(gap['person_login'])

    if not can_see_gap(gap, observer, person):
        return HttpResponseForbidden()

    return _dehydrate(gap)


@responding_json
@login_required
@require_GET
@available_for_external('gap.robot_with_gap_api_access')
@available_by_tvm(['maya'])
def gaps_find(request):
    yauser = request.yauser
    if isinstance(yauser, TvmServiceRequest):  # service already checked in decorator
        return HttpResponseForbidden()
    observer = request.user
    GET = request.GET

    person_logins = (
        GET.getlist('person_login') if 'person_login' in GET
        else [observer.get_profile().login]
    )

    # STAFF-17109
    if settings.LOGIN_TIGRAN in person_logins and observer.get_profile().login != settings.LOGIN_TIGRAN:
        person_logins.remove(settings.LOGIN_TIGRAN)

    workflows = GET.getlist('workflow') if 'workflow' in GET else None
    states = GET.getlist('state') if 'state' in GET else None
    sorting = GET.get('sorting', 'date_from')

    try:
        ordering = int(GET.get('ordering', 1))

        if ordering not in [-1, 1]:
            return HttpResponseBadRequest()

        limit = int(request.GET.get('limit', 100))
        page = int(request.GET.get('page', 0))

        if limit < 1 or limit > 1000:
            limit = 100
        if page < 0:
            page = 0

        date_from = dateutil.parser.parse(request.GET.get('date_from')).replace(
            tzinfo=None)
        date_to = dateutil.parser.parse(request.GET.get('date_to')).replace(
            tzinfo=None)
    except (TypeError, ValueError, AttributeError):
        return HttpResponseBadRequest()

    try:
        total = _gaps_find_total_count(person_logins, workflows, states, date_from, date_to)
        gaps = _gaps_find_paged(person_logins, workflows, states, date_from,
                                date_to, sorting, ordering, limit, page)
        _gaps = [_dehydrate(gap) for gap in gaps]
        return {
            'page': page,
            'pages': total // limit + (1 if total % limit else 0),
            'limit': limit,
            'total': total,
            'gaps': _gaps,
        }
    except Exception:
        return HttpResponseServerError()


def _gaps_find_query(person_logins, workflows, states, date_from, date_to):
    gqb = (
        GapQueryBuilder()
        .person_logins(list(set(person_logins)))
        .dates_not_strict(date_from, date_to)
        .workflows_states(LIMITED_STATES)
    )

    if workflows is not None:
        gqb.workflows(workflows)

    if states is not None:
        gqb.states(states)

    return gqb.query()


def _gaps_find_paged(person_logins, workflows, states, date_from, date_to,
                     sorting, ordering, limit, page):
    sorting = [(sorting, ordering)]

    gaps = GapCtl().find_gaps(
        query=_gaps_find_query(person_logins, workflows, states, date_from, date_to),
        sorting=sorting,
    )

    return gaps.skip(page * limit).limit(limit)


def _gaps_find_total_count(person_logins, workflows, states, date_from, date_to):
    return GapCtl().find_gaps(
        query=_gaps_find_query(person_logins, workflows, states, date_from, date_to),
    ).count()


def _dehydrate(gap):
    w_fields = WORKFLOW_FIELDS[gap['workflow']]
    return {
        k: v for k, v in gap.items()
        if k in w_fields
    }
