from datetime import datetime
import json
import pymongo

from django.conf import settings
from django.contrib.auth.decorators import login_required
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_http_methods

from staff.lib.utils.date import parse_datetime
from staff.lib.decorators import responding_json, available_for_external
from staff.gap.api.views.workflow_views import get_workflows_ref
from staff.gap.controllers.gap import GapCtl, GapQueryBuilder
from staff.gap.controllers.staff_utils import GapsByPersons
from staff.gap.workflows.choices import LIMITED_STATES


import logging
logger = logging.getLogger('staff.gap.api.views.export_gaps_views')


GAP_DEFAULT_FIELDS = [
    'id',
    'workflow',
    'comment',
    'date_from',
    'date_to',
    'work_in_absence',
    'full_day',
]

GAP_ALLOWED_FIELDS = [
    'id',
    'workflow',
    'person_login',
    'person_id',
    'comment',
    'date_from',
    'date_to',
    'full_day',
    'work_in_absence',
    'to_notify',
    'state',
]


@csrf_exempt
@require_http_methods(['GET', 'POST'])
@login_required
@responding_json
@available_for_external('gap.robot_with_gap_api_access')
def export_gaps_json(request):
    observer = request.user
    if request.method == 'GET':
        params = _gaps_params_get(request.GET, observer)
    else:
        params = _gaps_params_post(request.body, observer)

    return _export_gaps_json(**params)


def _export_gaps_json(gqb=None, fields=None, only_one=False):
    sorting = [
        ('person_login', pymongo.ASCENDING),
        ('date_from', pymongo.ASCENDING),
    ]
    persons = {}
    gaps = GapCtl().find_gaps(query=gqb.query(), fields=GAP_ALLOWED_FIELDS, sorting=sorting)
    for gap in gaps:
        _gaps = persons.setdefault(gap['person_login'], [])
        _gaps.append(gap)

    if only_one:
        for person_login, _gaps in persons.items():
            _gaps.sort(key=GapsByPersons._sort_key)
            persons[person_login] = _dehydrate_gaps([_gaps[0]], fields)
    else:
        for person_login, _gaps in persons.items():
            persons[person_login] = _dehydrate_gaps(_gaps, fields)

    return {
        'persons': persons,
        'workflows': get_workflows_ref(),
    }


def _dehydrate_gaps(gaps, fields):
    return [{k: v for k, v in gap.items() if k in fields} for gap in gaps]


def _gaps_params_get(params, observer):
    person_logins = params.getlist('l')
    # STAFF-17109
    if settings.LOGIN_TIGRAN in person_logins and observer.get_profile().login != settings.LOGIN_TIGRAN:
        person_logins.remove(settings.LOGIN_TIGRAN)

    workflows = params.getlist('workflow') if 'workflow' in params else None
    fields = _filter_fields(params.getlist('field')) if 'field' in params else GAP_DEFAULT_FIELDS
    only_one = bool(int(params.get('only_one', 0)))

    date_from = parse_datetime(params.get('date_from')) if 'date_from' in params else datetime.utcnow()
    if only_one:
        date_to = date_from
    else:
        date_to = parse_datetime(params.get('date_to')) if 'date_to' in params else date_from

    return {
        'gqb': _get_export_gqb(person_logins, workflows, date_from, date_to),
        'fields': fields,
        'only_one': only_one,
    }


def _gaps_params_post(body, observer):
    data = json.loads(body)

    workflows = data.get('workflows') if 'workflows' in data else None
    fields = _filter_fields(data.get('fields')) if 'fields' in data else GAP_DEFAULT_FIELDS
    person_logins = data.get('person_logins')
    # STAFF-17109
    if settings.LOGIN_TIGRAN in person_logins and observer.get_profile().login != settings.LOGIN_TIGRAN:
        person_logins.remove(settings.LOGIN_TIGRAN)

    only_one = data.get('only_one', False)

    date_from = parse_datetime(data.get('date_from')) if 'date_from' in data else datetime.utcnow()
    if only_one:
        date_to = date_from
    else:
        date_to = parse_datetime(data.get('date_to')) if 'date_to' in data else date_from

    return {
        'gqb': _get_export_gqb(person_logins, workflows, date_from, date_to),
        'fields': fields,
        'only_one': only_one,
    }


def _filter_fields(fields):
    return [field for field in fields if field in GAP_ALLOWED_FIELDS] or GAP_DEFAULT_FIELDS


def _get_export_gqb(person_logins, workflows, date_from, date_to):
    gqb = (
        GapQueryBuilder()
        .dates_not_strict(date_from, date_to)
        .workflows_states(LIMITED_STATES)
    )

    if person_logins:
        gqb.person_logins(person_logins)

    if workflows:
        gqb.workflows(workflows)

    return gqb
