import pymongo

from django.conf import settings
from django.contrib.auth.decorators import login_required
from django.views.decorators.http import require_GET
from django.core.urlresolvers import reverse

from staff.person.models import Staff
from staff.person_profile.controllers.subordinates import direct_subordinates_q
from staff.lib.decorators import responding_json, available_for_external
from staff.gap.controllers.gap import GapCtl, GapQueryBuilder


import logging
logger = logging.getLogger('staff.gap.api.views.need_for_attention_views')


@login_required
@require_GET
@responding_json
@available_for_external('gap.robot_with_gap_api_access')
def need_for_attention(request):
    observer = request.user
    observer_id = observer.get_profile().id
    direct_subordinates_ids = list(
        Staff.objects
        .filter(direct_subordinates_q(observer_id))
        .values_list('id', flat=True)
    )
    gaps = list(_get_unconfirmed_gaps(direct_subordinates_ids))
    persons_data = (
        Staff.objects.filter(
            id__in=set([gap['person_id'] for gap in gaps])
        ).values(
            'login',
            'first_name',
            'last_name',
            'first_name_en',
            'last_name_en',
        )
    )

    persons_dict = {p['login']: p for p in persons_data}

    for gap in gaps:
        gap['person'] = persons_dict[gap['person_login']]
        gap['confirm_url'] = 'https://%s%s' % (
            settings.STAFF_DOMAIN,
            reverse('gap:api-gap-confirm', kwargs={'gap_id': gap['id']}),
        )

        del gap['person_login']
        del gap['person_id']

    return {
        'unconfirmed_vacations': gaps,
    }


def _get_unconfirmed_gaps(persons_ids):
    query = (
        GapQueryBuilder()
        .person_ids(persons_ids)
        .state('new')
        .workflows(['vacation'])
        .query()
    )

    fields = [
        'id',
        'date_from',
        'date_to',
        'person_login',
        'person_id',
        'comment',
    ]

    sorting = [('date_from', pymongo.DESCENDING)]

    return GapCtl().find_gaps(query=query, fields=fields, sorting=sorting)
