import pymongo
from datetime import datetime, timedelta

from django.contrib.auth.decorators import login_required
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_http_methods

from staff.person.models import Staff
from staff.lib.utils.date import parse_datetime
from staff.lib.decorators import responding_csv, available_for_external

from staff.gap.controllers.gap import GapCtl, GapQueryBuilder
from staff.gap.workflows.choices import GAP_STATES as GS

import logging
logger = logging.getLogger('staff.gap.api.views.readable_gaps_views')


@csrf_exempt
@require_http_methods(['GET'])
@login_required
@responding_csv(filename='gaps')
@available_for_external('gap.robot_with_gap_api_access')
def readable_gaps(request):
    date_from = parse_datetime(request.GET.get('date_from')) if 'date_from' in request.GET else datetime.utcnow()
    date_to = parse_datetime(request.GET.get('date_to')) if 'date_to' in request.GET else date_from

    gaps = _get_gaps(date_from, date_to)
    persons = _collect_persons(gaps)

    return _merge_data(gaps, persons)


def _get_gaps(date_from, date_to):
    gqb = (
        GapQueryBuilder()
        .dates_not_strict(date_from, date_to)
        .full_day(True)
        .workflow('absence')
        .state(GS.NEW)
    )

    fields = [
        'person_login',
        'person_id',
        'date_from',
        'date_to',
    ]

    sorting = [
        ('person_login', pymongo.ASCENDING),
        ('date_from', pymongo.ASCENDING),
    ]

    return list(GapCtl().find_gaps(query=gqb.query(), fields=fields, sorting=sorting))


def _collect_persons(gaps):
    persons_ids = set(g['person_id'] for g in gaps)
    persons = Staff.objects.filter(id__in=persons_ids).values('id', 'first_name', 'last_name')
    return {p['id']: p for p in persons}


def _merge_data(gaps, persons):
    result = []
    for gap in gaps:
        person_id = gap['person_id']
        person = persons[person_id]
        result.append((
            '%s %s' % (person['first_name'], person['last_name']),
            gap['person_login'],
            '%s - %s' % (gap['date_from'].date(), gap['date_to'].date() - timedelta(days=1)),
            (gap['date_to'].date() - gap['date_from'].date()).days,
        ))

    return result
