from datetime import datetime

from staff.lib.utils.ordered_choices import OrderedChoices

from staff.gap.controllers.mongo import MongoCtl


EMAIL_TO_SEND_STATES = OrderedChoices(
    ('NEW', 'new'),
    ('REQUESTED', 'requested'),
    ('WAS_SENT', 'was_sent'),
    ('ERROR', 'error'),
)


class EmailToSendCtl(MongoCtl):

    MONGO_COLLECTION = 'email_to_send'

    def __init__(self):
        pass

    def new(self, gap_id, recipients, text, workflow, tag):
        now = datetime.utcnow()
        data = {
            'gap_id': gap_id,
            'recipients': recipients,
            'text': text,
            'state': EMAIL_TO_SEND_STATES.NEW,
            'created_at': now,
            'modified_at': now,
            'workflow': workflow,
            'tag': tag,
        }
        self._collection().insert_one(data)
        return data['_id']

    def find_by_state(self, state):
        query = {
            'state': state,
        }
        fields = [
            '_id',
            'gap_id',
            'recipients',
            'text',
            'state',
        ]
        return self.find(query=query, fields=fields)

    def find_to_send(self, set_requested=False):
        if set_requested:
            result = list(self.find_by_state(EMAIL_TO_SEND_STATES.NEW))
            _ids = [email['_id'] for email in result]
            self.set_state_many(_ids, EMAIL_TO_SEND_STATES.REQUESTED)
            return result
        return list(self.find_by_state(EMAIL_TO_SEND_STATES.NEW))

    def set_state_one(self, _id, state):
        self._collection().update(
            {'_id': _id},
            {'$set': {
                'state': state,
                'modified_at': datetime.utcnow(),
            }},
        )

    def set_state_many(self, _ids, state):
        self._collection().update(
            {'_id': {'$in': _ids}},
            {'$set': {
                'state': state,
                'modified_at': datetime.utcnow(),
            }},
        )
