from django.conf import settings

from staff.lib.requests import get_ids_repository
from staff.lib.startrek.issues import add_comment, change_state, get_issue, get_issues, get_issues_by_unique

import logging
logger = logging.getLogger('staff.gap.controllers.startrek')


class StartrekCtlException(Exception):
    pass


class StartrekCtl:
    def __init__(self, author_login=None, queue=None, issue_key=None):
        self.queue = queue
        self.issue_key = issue_key
        self.author_login = author_login
        self._issue = None
        self._ids_repository = None

    @property
    def issue(self):
        if self.issue_key and not self._issue:
            self._issue = get_issue(self.issue_key)

            if not self._issue:
                raise StartrekCtlException(f'Issue {self.issue_key} not found')

        return self._issue

    def find_by_keys(self, keys):
        return get_issues(keys)

    def find_by_unique(self, unique):
        return get_issues_by_unique(unique)

    @property
    def ids_repository(self, force=False):
        if not self._ids_repository or force:
            self._ids_repository = get_ids_repository(
                'startrek2',
                'issues',
                user_agent='Gap ST client',
                oauth_token=settings.ROBOT_STAFF_OAUTH_TOKEN,
            )
        return self._ids_repository

    def create(self, **params):
        if 'createdBy' not in params:
            params['createdBy'] = self.author_login
        return self.ids_repository.create(**params)

    def comment(self, text, summonees=None):
        if self.author_login is not None:
            text = '@{login}:\n<[{text}]>'.format(login=self.author_login, text=text)

        assert self.issue_key
        add_comment(self.issue_key, text, summonees=summonees)

    def change_state(self, action_id):
        assert self.issue_key
        assert action_id
        change_state(self.issue_key, action_id)

    def get_available_states(self):
        return [transition.id for transition in getattr(self.issue, 'transitions', [])]
