from copy import deepcopy
from datetime import timedelta, datetime
from pytz import timezone
from typing import Dict, Optional

from django.conf import settings
from django.core.urlresolvers import reverse

from django.utils.translation import override

from staff.person.models import Staff

from staff.gap.workflows.utils import find_workflow
from staff.gap.controllers.query_builder import QueryBuilder
from staff.gap.controllers.mongo import MongoCtl
from staff.gap.controllers.gap import GapCtl
from staff.gap.controllers.utils import (
    get_gap_datetimes,
    datetime_from_utc,
    map_weekdays_to_words,
    periodic_gap_date_to,
)

import logging
logger = logging.getLogger('staff.gap.controllers.templates')


TEMPLATE_FILTER_FIELDS = [
    'login',
    'lang_ui',
    'organization_id',
    'office__city_id',
]

PERSON_CONTEXT_FIELDS = []

ALL_FIELDS = TEMPLATE_FILTER_FIELDS + PERSON_CONTEXT_FIELDS


class TemplatesCtl(MongoCtl):

    MONGO_COLLECTION = 'templates'

    filter_fields = [
        'workflow',
        'type',
        'lang',
        'tag',
        'organization_id',
        'city_id',
    ]

    def __init__(self, cached=False):
        self.cached = cached
        if self.cached:
            self.cache = {}

    def _create_query(self, **kwargs):
        tqb = TemplateQueryBuilder()
        for field_name in self.filter_fields:
            getattr(tqb, field_name)(kwargs.get(field_name))
        return tqb.query()

    def new(self, **kwargs):
        data = {
            'title': kwargs.get('title', ''),
            'type': kwargs['type'],
            'workflows': kwargs.get('workflows'),
            'tag': kwargs['tag'],
            'organization_ids': kwargs.get('organization_ids'),
            'city_ids': kwargs.get('city_ids'),
            'langs': kwargs.get('langs'),
            'template': kwargs['template'],
        }
        self._collection().insert_one(data)
        return data['_id']

    def update(self, data):
        self._collection().update_one({'_id': data['_id']}, {'$set': data})

    def find_one_strict(self, **kwargs) -> Optional[Dict]:
        return self.find_one(self._create_query(**kwargs))

    def find_one_not_strict(self, **kwargs):
        result = self.find_one_strict(**kwargs)
        if result:
            return result
        weak_filter = kwargs.copy()
        weak_order = [
            'city_id',
            'organization_id',
            'workflow',
            'lang',
        ]
        for field in weak_order:
            if field in weak_filter and weak_filter[field] is not None:
                weak_filter['weak_%s' % field] = weak_filter[field]
                del weak_filter[field]
                result = self.find_one_strict(**weak_filter)
                if result:
                    return result

    def find_one(self, query, del_id=True):
        if self.cached:
            return self._find_one_cached(query)
        return self._collection().find_one(filter=query)

    def _find_one_cached(self, query):
        q_hash = self.query_hash(query)
        if q_hash in self.cache:
            return self.cache[q_hash]
        result = self._collection().find_one(filter=query)
        self.cache[q_hash] = deepcopy(result)
        return result

    def query_hash(self, query):
        return hash(str(query))


class TemplateQueryBuilder(QueryBuilder):

    def __init__(self):
        super(TemplateQueryBuilder, self).__init__()

    def workflow(self, workflow):
        if workflow is None:
            return self.empty_workflow()
        return self.op_eq('workflows', workflow)

    def empty_workflow(self):
        return self.op_eq('workflows', None)

    def weak_workflow(self, workflow):
        return self.op_or([
            {'workflows': workflow},
            {'workflows': None}],
        )

    def type(self, type):
        return self.op_eq('type', type)

    def lang(self, lang):
        if lang is None:
            return self.empty_lang()
        return self.op_eq('langs', lang)

    def empty_lang(self):
        return self.op_eq('langs', None)

    def weak_lang(self, lang):
        return self.op_or([
            {'langs': lang},
            {'langs': None}],
        )

    def tag(self, tag):
        return self.op_eq('tag', tag)

    def organization_id(self, organization_id):
        if organization_id is None:
            return self.empty_organization_id()
        return self.op_eq('organization_ids', organization_id)

    def empty_organization_id(self):
        return self.op_eq('organization_ids', None)

    def weak_organization_id(self, organization_id):
        return self.op_or([
            {'organization_ids': organization_id},
            {'organization_ids': None}],
        )

    def city_id(self, city_id):
        if city_id is None:
            return self.empty_city_id()
        return self.op_eq('city_ids', city_id)

    def empty_city_id(self):
        return self.op_eq('city_ids', None)

    def weak_city_id(self, city_id):
        return self.op_or([
            {'city_ids': city_id},
            {'city_ids': None}],
        )


def get_templates_by_logins(logins, type, workflow, tag) -> Dict:
    t_ctl = TemplatesCtl(cached=True)
    q = Staff.objects.filter(login__in=logins).values(*ALL_FIELDS)
    result = {}
    for person in q:
        login = person['login']
        template = t_ctl.find_one_not_strict(
            type=type,
            workflow=workflow,
            tag=tag,
            lang=person['lang_ui'] or 'ru',
            organization_id=person['organization_id'],
            city_id=person['office__city_id'],
        )
        result[login] = {
            'context': {
                'addressee': person,
            },
            'template': template,
        }
    return result


class TemplateContext(object):

    person_fields = [
        'id',
        'login',
        'first_name',
        'last_name',
        'first_name_en',
        'last_name_en',
        'tz',
        'organization__city__country__name',
        'organization__city__country__name_en',
        'organization__name',
        'organization__name_en',
        'organization__st_translation_id',
    ]

    modifier_fields = [
        'id',
        'login',
        'first_name',
        'last_name',
        'first_name_en',
        'last_name_en',
        'organization__city__country__name',
        'organization__city__country__name_en',
        'organization__name',
        'organization__name_en',
        'organization__st_translation_id',
    ]

    workflow_fields = [
        'workflow',
        'verbose_name',
        'verbose_name_en',
        'new_gap_head_ru',
        'new_gap_head_en',
        'edit_gap_head_ru',
        'edit_gap_head_en',
        'cancel_gap_head_ru',
        'cancel_gap_head_en',
    ]

    datetime_fields = [
        'date_from_year',
        'date_from_day_year',
        'date_from_day',
        'date_from_time',
        'date_from_day_time',
        'date_from_full',
        'date_to_year',
        'date_to_day_year',
        'date_to_day',
        'date_to_time',
        'date_to_day_time',
        'date_to_full',
        'date_range',
    ]

    _context = {}

    def __init__(self, person, modifier):
        self.person = person
        self.modifier = modifier
        self._context = {}
        self._person()
        self._modifier()

    def _convert(self, prefix, data, allowed_fields, postfix=''):
        return {
            '%s%s%s' % (prefix, key.replace('__', '_'), postfix): value
            for key, value in data.items()
            if key in allowed_fields
        }

    def _inflection(self, prefix, person, form_name):
        fields = ['first_name', 'last_name']
        with override('ru'):
            staff = Staff.objects.get(id=person['id'])
            inflection = staff.inflections.inflect(form_name).split(' ')

        return {
            f'{prefix}{field}_{form_name}': inflection[index]
            for index, field in enumerate(fields)
        }

    def _workflow(self, workflow):
        workflow_cls = find_workflow(workflow)
        workflow_context = {
            'workflow': workflow_cls.workflow,
            'verbose_name': workflow_cls.verbose_name,
            'verbose_name_en': workflow_cls.verbose_name_en,
        }
        workflow_context.update(workflow_cls.template_context)

        self._context.update(
            self._convert('workflow_', workflow_context, self.workflow_fields)
        )

        return self

    def _person(self):
        self._context.update(self._convert('person_', self.person, self.person_fields))
        self._context.update(self._inflection('person_', self.person, 'genitive'))

        return self

    def _modifier(self):
        self._context.update(self._convert('modifier_', self.modifier, self.modifier_fields))

        return self

    def _hactions(self, hactions, _id):
        if not hactions:
            return
        for action_hash, haction_data in hactions.items():
            haction_url = 'https://%s%s' % (
                settings.STAFF_DOMAIN,
                reverse('gap:api-haction', args=[action_hash, _id]),
            )
            haction_name = haction_data['haction_name']
            self._context['haction_url_%s' % haction_name] = haction_url

    def _gap_dates(self, date_from, date_to, prefix, lang, full_day):
        date_from, date_to = self._shift_gap_dates(date_from, date_to, full_day)
        dates = get_gap_datetimes(date_from, date_to, full_day, lang)
        self._context.update(self._convert(prefix, dates, self.datetime_fields, postfix='_' + lang))

    def _shift_gap_dates(self, date_from, date_to, full_day):
        if not full_day:
            person_tz = timezone(self.person['tz'])
            return (
                datetime_from_utc(date_from, person_tz),
                datetime_from_utc(date_to, person_tz),
            )
        else:
            return (
                date_from,
                date_to - timedelta(days=1),
            )

    def context(self):
        return self._context


class GapTemplateContext(TemplateContext):

    gap_fields = [
        'id',
        'full_day',
        'work_in_absence',
        'comment',
        'to_notify',
        'is_selfpaid',
        'has_sicklist',
        'is_covid',
        'mandatory',
    ]

    gap_edit_fields = [
        'date_from',
        'date_to',
        'full_day',
        'work_in_absence',
        'comment',
        'to_notify',
        'is_selfpaid',
        'has_sicklist',
        'is_covid',
    ]

    def __init__(self, person, modifier, data, data_diff=None):
        super().__init__(person, modifier)

        self.gap_data = data
        self._gap()
        self._hactions(
            data.get('hastions'),
            data['id'],
        )
        self._workflow(data['workflow'])

        self.gap_diff = data_diff
        if data_diff:
            self._gap_diff()

    def _gap(self):
        gap = self.gap_data
        self._context.update(self._convert('gap_', gap, self.gap_fields))
        self._gap_dates(gap['date_from'], gap['date_to'], 'gap_', 'ru', gap['full_day'])
        self._gap_dates(gap['date_from'], gap['date_to'], 'gap_', 'en', gap['full_day'])

        return self

    def _gap_diff(self):
        if not self.gap_diff:
            return
        gap_diff = self.gap_diff

        for field in self.gap_edit_fields:
            self._context['gap_%s_changed' % field] = field in gap_diff

        for field, values in gap_diff.items():
            if field in self.gap_fields and field != 'date_from' and field != 'date_to':
                self._context['gap_old_%s' % field] = values['old']
                self._context['gap_new_%s' % field] = values['new']

        if 'date_from' in gap_diff and 'date_to' in gap_diff:
            new_full_day = self.gap_data['full_day']
            old_full_day = new_full_day
            old_date_from = gap_diff['date_from']['old']
            old_date_to = gap_diff['date_to']['old']
            if 'full_day' in self.gap_diff:
                old_full_day = gap_diff['full_day']['old']
            self._gap_dates(old_date_from, old_date_to, 'gap_old_', 'ru', old_full_day)
            self._gap_dates(old_date_from, old_date_to, 'gap_old_', 'en', old_full_day)
            self._gap_dates(gap_diff['date_from']['new'], gap_diff['date_to']['new'], 'gap_new_', 'ru', new_full_day)
            self._gap_dates(gap_diff['date_from']['new'], gap_diff['date_to']['new'], 'gap_new_', 'en', new_full_day)

        return self


class PeriodicGapTemplateContext(TemplateContext):

    periodic_gap_fields = [
        'id',
        'comment',
        'to_notify',
    ]

    def __init__(self, person, modifier, data, data_diff=None):
        super().__init__(person, modifier)

        self.periodic_gap_data = data
        self._periodic_gap()
        self._periodic_gap_weekdays(data['periodic_map_weekdays'])
        self._periodic_gap_date_to(data['periodic_date_to'])
        self._hactions(
            data.get('hactions'),
            data['id'],
        )
        self._workflow(data['workflow'])
        self._closest_gap_in_periodic_gap()

    def _periodic_gap(self):
        periodic_gap = self.periodic_gap_data
        self._context.update(self._convert('periodic_gap_', periodic_gap, self.periodic_gap_fields))
        self._gap_dates(periodic_gap['date_from'], periodic_gap['date_to'], 'closest_gap_', 'ru', full_day=True)
        self._gap_dates(periodic_gap['date_from'], periodic_gap['date_to'], 'closest_gap_', 'en', full_day=True)

        return self

    def _periodic_gap_weekdays(self, weekdays):
        self._context['periodic_gap_repeat_days_ru'] = map_weekdays_to_words(weekdays, 'ru')
        self._context['periodic_gap_repeat_days_en'] = map_weekdays_to_words(weekdays, 'en')

    def _periodic_gap_date_to(self, date):
        if self.periodic_gap_data['date_from'] <= date:
            self._context['periodic_gap_date_to_ru'] = periodic_gap_date_to(date, 'ru')
            self._context['periodic_gap_date_to_en'] = periodic_gap_date_to(date, 'en')
        else:
            self._context['periodic_gap_date_to_ru'] = None
            self._context['periodic_gap_date_to_en'] = None

    def _closest_gap_in_periodic_gap(self):
        gap_ctl = GapCtl()

        next_gaps_cursor = gap_ctl.find({
            'date_from': {
                '$gte': datetime.now() - timedelta(days=1),
                '$lte': self.periodic_gap_data['periodic_date_to'],
            },
            'periodic_gap_id': self.periodic_gap_data['id'],
        })

        closest_gap = None
        if next_gaps_cursor.count() > 0:
            closest_gap = next(next_gaps_cursor)
        else:
            previous_gaps_cursor = gap_ctl.find({
                'periodic_gap_id': self.periodic_gap_data['id'],
                'date_from': {
                    '$lte': self.periodic_gap_data['periodic_date_to'],
                },
            }).sort([('_id', -1)])
            if previous_gaps_cursor.count() > 0:
                closest_gap = next(previous_gaps_cursor)

        if closest_gap is not None:
            self._context['closest_gap_id'] = closest_gap['id']
        else:
            self._context['closest_gap_id'] = None
