import locale
from datetime import timedelta, datetime
from pytz import timezone
from typing import List, Dict

from django.core.validators import validate_email
from django.core.exceptions import ValidationError
from django.conf import settings

from staff.lib.requests import get_ids_repository
from staff.lib.models.roles_chain import chiefs_chain_for_person
from staff.lib.utils.qs_values import extract_related, localize

from staff.departments.models import DepartmentStaff, DepartmentRoles
from staff.departments.utils import get_department_chain

from staff.person.models import Staff, DOMAIN
from staff.person_profile.errors import log_does_not_exist_staff_login
from staff.maillists.models import List as Maillist

from staff.gap.exceptions import GapViewError

import logging
logger = logging.getLogger('staff.gap.controller.utils')


def get_chief(login: str, chief_fields: List[str] = None) -> Dict:
    chief_fields = chief_fields or ['login', 'work_email']
    person = Staff.objects.get(login=login)
    result = chiefs_chain_for_person(person, chief_fields)
    return result[0] if result else {}


def get_chiefs(login, chief_fields=None):
    chief_fields = chief_fields or ['login']
    dep = (
        Staff.objects
        .values(
            'department__lft',
            'department__rght',
            'department__tree_id',
            'department__id',
        ).get(login=login)
    )

    dep_ids = [d['id'] for d in get_department_chain(
        **extract_related(dep, 'department'))]

    try:
        chief_ids = list(
            DepartmentStaff.objects
            .filter(
                department__in=dep_ids,
                role_id=DepartmentRoles.CHIEF.value,
            )
            .exclude(staff__login=login)
            .order_by('-department__lft')
            .values_list('staff', flat=True)
        )
    except IndexError:
        return []

    return (
        Staff.objects
        .values(*chief_fields)
        .filter(id__in=chief_ids)
    )


def is_chief_of(observer_login, person_login):
    for chief in get_chiefs(person_login):
        if chief['login'] == observer_login:
            return True
    return False


def get_extended_localized_person(login):
    with log_does_not_exist_staff_login(logger=logger, message_params=[login], raise_e=GapViewError):
        return localize(Staff.objects.values(
            'id',
            'login',
            'first_name',
            'last_name',
            'first_name_en',
            'last_name_en',
            'office_id',
            'organization__city__country_id',
            'organization__country_code',
            'vacation',
            'paid_day_off',
            'is_homeworker',
            'work_mode',
            'room_id',
        ).get(login=login))


def get_short_person_with_department(login):
    return (Staff.objects.values(
        'id',
        'login',
        'organization_id',
        'department__tree_id',
        'department__lft',
        'department__rght',
        'work_mode',
    ).get(login=login))


def get_short_persons_with_department(logins):
    return (Staff.objects.values(
        'id',
        'login',
        'department__tree_id',
        'department__lft',
        'department__rght',
        'work_mode',
    ).filter(login__in=logins))


def get_universal_person(person_id=None, person_login=None):
    q = {'id': person_id} if person_id is not None else {'login': person_login}

    person_data = Staff.objects.values(
        'id',
        'uid',
        'login',
        'first_name',
        'last_name',
        'first_name_en',
        'last_name_en',
        'office_id',
        'office__name',
        'office__city_id',
        'organization_id',
        'organization__name',
        'organization__name_en',
        'organization__st_translation_id',
        'organization__city__country__name',
        'organization__city__country__name_en',
        'department__name',
        'department__lft',
        'department__rght',
        'department__tree_id',
        'department__id',
        'room_id',
        'work_email',
        'lang_ui',
        'tz',
    ).get(**q)

    person_data['is_ruby_crown_owner'] = DepartmentStaff.objects.filter(
        role_id=DepartmentRoles.GENERAL_DIRECTOR.value,
        staff_id=person_data['id'],
    ).exists()

    return person_data


def dehydrate_person(person):
    return {
        'id': person['id'],
        'login': person['login'],
        'first_name': person['first_name'],
        'last_name': person['last_name'],
        'work_mode': person['work_mode'],
    }


def is_gap_partial(date_from, date_to):
    return (
        date_from.year == date_to.year and
        date_from.month == date_to.month and
        date_from.day == date_to.day and
        (date_to - date_from).total_seconds() < 86400
    )


def get_gap_days(date_from, date_to, full_day, partial_as_full=False):
    if is_gap_partial(date_from, date_to):
        return 1 if partial_as_full else 0

    return (date_to.date() - date_from.date()).days + (0 if full_day else 1)


def get_full_vacation_days(date_from, date_to):
    return (date_to - date_from).days + 1


def replace_useless_smt(dt):
    return dt.replace(
        second=0,
        microsecond=0,
        tzinfo=None,
    )


def replace_useless_hmst(dt):
    return dt.replace(
        hour=0,
        minute=0,
        second=0,
        microsecond=0,
        tzinfo=None
    )


def full_day_dates(data, delta):
    data['date_from'] = data['date_from'].replace(
        hour=0,
        minute=0,
        second=0,
        microsecond=0,
    )
    data['date_to'] = data['date_to'].replace(
        hour=0,
        minute=0,
        second=0,
        microsecond=0,
    ) + timedelta(days=delta)


YA_DOMAINS = {d[1] for d in DOMAIN}


def recognize_emails(emails):
    all_emails = {
        email.strip() for email in emails if email
    } if emails else set()

    result = {
        'invalid': set(),
        'staff': set(),
        'ml': set(),
        'unknown': set(),
    }

    if not all_emails:
        return result

    valid_emails = set()

    # Невалидные адреса

    for email in all_emails:
        try:
            validate_email(email)
        except ValidationError:
            result['invalid'].add(email)
        else:
            valid_emails.add(email)

    if not valid_emails:
        return result

    pairs = {tuple(email.split('@')) for email in valid_emails}
    ya_domain_pairs = {pair for pair in pairs if pair[1] in YA_DOMAINS}

    # Адреса внешнего мира

    pairs -= ya_domain_pairs

    result['unknown'] = {'@'.join(pair) for pair in pairs}

    if not ya_domain_pairs:
        return result

    # Адреса сотрудников

    logins = {pair[0] for pair in ya_domain_pairs}

    staff = set(
        Staff.objects.filter(login__in=logins).values_list('login', flat=True)
    )

    staff_pairs = {pair for pair in ya_domain_pairs if pair[0] in staff}

    ya_domain_pairs -= staff_pairs

    result['staff'] = {(pair[0], '@'.join(pair)) for pair in staff_pairs}

    if not ya_domain_pairs:
        return result

    # Адреса рассылок

    remains = {'@'.join(pair) for pair in ya_domain_pairs}

    ml_emails = set(
        Maillist.objects.filter(email__in=remains).values_list('email', flat=True)
    )

    remains -= ml_emails

    result['ml'] = ml_emails

    # Остаток внешних адресов

    if remains:
        result['unknown'] = result['unknown'] | remains

    return result


def recognize_emails_for_ui(emails):
    result = []

    if not emails:
        return result

    logins_by_email = {}
    logins = set()

    for email in emails:
        try:
            validate_email(email)
        except ValidationError:
            pass
        else:
            parts = email.split('@')
            if parts[1] in YA_DOMAINS:
                login = parts[0]
                logins.add(login)
                logins_by_email[email] = login

    persons = (
        Staff.objects.filter(login__in=logins).values(
            'login',
            'first_name',
            'last_name',
            'first_name_en',
            'last_name_en',
        )
    )

    staff_by_login = {person['login']: localize(person) for person in persons}

    for email in emails:
        caption = email
        if email in logins_by_email:
            login = logins_by_email[email]
            if login in staff_by_login:
                staff = staff_by_login[login]
                caption = '%s %s' % (staff['first_name'], staff['last_name'])
        result.append({'email': (email, caption)})

    return result


def get_gap_datetimes(date_from, date_to, full_day, lang='ru'):
    current_locale = locale.getlocale(locale.LC_TIME)
    if lang == 'ru':
        locale.setlocale(locale.LC_TIME, ('ru', 'UTF8'))
    else:
        locale.setlocale(locale.LC_TIME, ('en', 'UTF8'))

    dates = {}
    dates.update(_convert_date(date_from, 'date_from_'))
    dates.update(_convert_date(date_to, 'date_to_'))

    diff = date_to - date_from

    if diff.days == 0:
        if full_day:
            result = '{date_from_day}'.format(**dates)
        else:
            result = '{date_from_day}, {date_from_time} - {date_to_time}'.format(**dates)
    elif date_from.year == date_to.year:
        if full_day:
            result = '{date_from_day} - {date_to_day}'.format(**dates)
        else:
            result = '{date_from_day_time} - {date_to_day_time}'.format(**dates)
    else:
        if full_day:
            result = '{date_from_day_year} - {date_to_day_year}'.format(**dates)
        else:
            result = '{date_from_full} - {date_to_full}'.format(**dates)

    locale.setlocale(locale.LC_TIME, current_locale)

    dates['date_range'] = result

    return dates


RU_MONTH = {
    'Январь': 'января',
    'Февраль': 'февраля',
    'Март': 'марта',
    'Апрель': 'апреля',
    'Май': 'мая',
    'Июнь': 'июня',
    'Июль': 'июля',
    'Август': 'августа',
    'Сентябрь': 'сентября',
    'Октябрь': 'октября',
    'Ноябрь': 'ноября',
    'Декабрь': 'декабря',
}


def _convert_date(gap_date, prefix):
    date_year = gap_date.year
    month = gap_date.strftime('%B')
    date_day = '%s %s' % (gap_date.day, RU_MONTH.get(month, month))
    date_day_year = '%s %s' % (date_day, date_year)
    date_time = gap_date.strftime('%H:%M')
    date_day_time = '%s %s' % (date_day, date_time)
    date_full = '%s %s %s' % (date_day, date_time, date_year)
    return {
        prefix + 'year': date_year,
        prefix + 'day_year': date_day_year,
        prefix + 'day': date_day,
        prefix + 'time': date_time,
        prefix + 'day_time': date_day_time,
        prefix + 'full': date_full,
    }


def get_unique_issues(gap):
    result = []
    master = gap['master_issue'] if 'master_issue' in gap else ''
    if master:
        result.append(master)
    slave = gap['slave_issues'] if 'slave_issues' in gap else []
    if slave:
        result.extend([issue for issue in slave if issue != master])
    return result


_UTC_TZ = timezone('UTC')


def gap_dates_from_utc(gap, dst_tz):
    if not gap['full_day']:
        gap['date_from'] = datetime_from_utc(gap['date_from'], dst_tz)
        gap['date_to'] = datetime_from_utc(gap['date_to'], dst_tz)


def datetime_from_utc(date_time, dst_tz):
    return _UTC_TZ.localize(date_time).astimezone(dst_tz).replace(tzinfo=None)


def datetime_to_utc(date_time, src_tz=None):
    if src_tz and not date_time.tzinfo:
        date_time = src_tz.localize(date_time)
    return date_time.astimezone(_UTC_TZ).replace(tzinfo=None) if date_time.tzinfo else date_time


def collect_confirmed_by(gaps):
    result = {}
    person_ids = {gap['confirmed_by_id'] for gap in gaps if gap.get('confirmed_by_id')}

    if not person_ids:
        return {}

    inflector = get_ids_repository('inflector', 'inflector', user_agent=settings.STAFF_USER_AGENT)

    persons = Staff.objects.values(
        'id',
        'login',
        'first_name',
        'last_name',
        'first_name_en',
        'last_name_en',
        'gender',
    ).filter(id__in=person_ids)

    for person in persons:
        person = localize(person)
        person_data = result.setdefault(person['id'], {'login': person['login']})
        person['gender'] = person['gender'].lower()

        try:
            inflected = inflector.inflect_person(person, 'творительный')
        except Exception:
            inflected = person

        person_data['name_to_show'] = '%s %s' % (inflected['first_name'], inflected['last_name'])

    return result


WEEKDAYS_ACCUSATIVE = {
    'ru': [
        'понедельник',
        'вторник',
        'среду',
        'четверг',
        'пятницу',
        'субботу',
        'воскресенье',
    ],
    'en': [
        'monday',
        'tuesday',
        'wednesday',
        'thursday',
        'friday',
        'saturday',
        'sunday',
    ],
}

AND_WORD = {
    'ru': 'и',
    'en': 'and',
}


def map_weekdays_to_words(map_weekdays: list, lang='ru') -> str:
    map_weekdays_words = [WEEKDAYS_ACCUSATIVE[lang][day_num] for day_num in map(int, map_weekdays)]
    res = ', '.join(map_weekdays_words[:-1])
    if len(map_weekdays) > 1:
        res += f' {AND_WORD[lang]} '

    res += map_weekdays_words[-1]

    if lang == 'ru':
        if map_weekdays_words[0] == 'вторник':
            res = 'во ' + res
        else:
            res = 'в ' + res

    return res


def periodic_gap_date_to(date: datetime, lang='ru') -> str:
    if lang == 'ru':
        locale.setlocale(locale.LC_TIME, ('ru', 'UTF8'))
        month = date.strftime('%B')
        return f'{date.day} {RU_MONTH.get(month, month)}'
    else:
        locale.setlocale(locale.LC_TIME, ('en', 'UTF8'))
        return date.strftime('%B %d')
