import json
import logging
import sform

from django.conf import settings
from django.views.decorators.http import require_http_methods
from django.http import (
    HttpResponseNotFound,
    HttpResponseServerError,
    HttpResponseForbidden,
    JsonResponse,
)

from staff.lib.decorators import responding_json, available_for_external
from staff.lib.forms.errors import invalid_json_error
from staff.lib.log import log_context
from staff.lib.utils.qs_values import localize
from staff.person_filter.saved_filter_ctl import get_subscribers
from staff.person.models import Staff

from staff.gap.controllers.utils import (
    get_chief,
    get_extended_localized_person,
    full_day_dates,
    recognize_emails_for_ui,
    get_unique_issues,
    dehydrate_person,
)
from staff.gap.permissions import can_see_issues, can_see_gap, get_accessible_logins
from staff.gap.workflows import TRIP_WORKFLOWS
from staff.gap.workflows.form_factories import workflow_form_factory
from staff.gap.controllers.gap import GapCtl
from staff.gap.controllers.staff_utils import collect_forms_person_ids
from staff.gap.exceptions import GapError, MandatoryVacationError
from staff.gap.workflows.utils import find_workflow, find_form, find_workflows_by_lang


logger = logging.getLogger('staff.gap.edit_views.edit_gap_view')

FORM_NAME = 'gap'


@responding_json
@require_http_methods(['GET', 'POST'])
@available_for_external
def edit_gap(request, login, gap_id=None):
    observer = request.user

    if not get_accessible_logins(observer.get_profile(), [login]):
        logger.info('%s has no access to %s gap', observer, login)
        return HttpResponseForbidden()

    try:
        person = get_extended_localized_person(login)
    except GapError:
        return HttpResponseNotFound()

    person_id = person['id']

    modifier_id = observer.get_profile().id

    if request.method == 'GET':
        total_subscribers_count = get_subscribers(person_id, 'absences').count()

        subscriptions = {'subscribers_count': total_subscribers_count}

        chief = localize(get_chief(login, ['login', 'first_name', 'last_name', 'first_name_en', 'last_name_en']))

        if chief:
            subscriptions['chief'] = chief

        if gap_id is None:
            return _new_gap_get(modifier_id, person, subscriptions)
        else:
            return _edit_gap_get(person, gap_id, subscriptions, observer)

    try:
        form_data = json.loads(request.body)
    except ValueError:
        return invalid_json_error(request.body), 400

    if gap_id is None:
        workflow = form_data['workflow']
        gap = None
        cur_gap_days = 0
    else:
        try:
            gap = GapCtl().find_gap_by_id(gap_id)
        except GapError:
            return HttpResponseNotFound()

        cur_gap_days = (gap['date_to'] - gap['date_from']).days
        workflow = gap['workflow']

    try:
        workflow_cls = find_workflow(workflow)
        form_cls = find_form(workflow)
    except KeyError:
        return {
            'errors': {
                'workflow': [{'code': 'required'}],
            }
        }

    # костылище (c) another
    if workflow not in ['absence', 'learning']:
        form_data['full_day'] = True

    form = form_cls(data=form_data)

    if workflow == 'paid_day_off':
        valid_gap_len_days = (person['paid_day_off'] or 0) + cur_gap_days
        cant_see_paid_day_off = (
            person['organization__country_code'] != settings.RUSSIA_CODE or
            valid_gap_len_days < 1
        )
        if cant_see_paid_day_off:
            form.valid_gap_length_days = 0
        else:
            form.valid_gap_length_days = valid_gap_len_days

    if not form.is_valid():
        return form.errors

    cleaned_data = form.cleaned_data

    # костылим, чтобы SForm поддержать
    if 'to_notify' in cleaned_data:
        cleaned_data['to_notify'] = [email['email'] for email in cleaned_data['to_notify']]

    if cleaned_data['full_day']:
        full_day_dates(cleaned_data, 1)

    try:
        if gap:
            with log_context(gap_id=gap['id'], expected_exceptions=[MandatoryVacationError]):
                workflow_cls.init_to_modify(modifier_id, gap=gap).edit_gap(cleaned_data)
        elif cleaned_data.get('is_periodic_gap'):
            gap_id = workflow_cls.init_to_new(modifier_id, person_id).new_periodic_gap(cleaned_data)['id']
        else:
            gap_id = workflow_cls.init_to_new(modifier_id, person_id).new_gap(cleaned_data)['id']
    except MandatoryVacationError as err:
        return JsonResponse(err.error_dict, status=400)
    except Exception:
        logger.exception('Error modifying gap[%s]' % gap_id)
        return HttpResponseServerError()

    return {'id': gap_id}


def _edit_gap_get(person, gap_id, subscriptions, observer):
    observer_id = observer.get_profile().id
    try:
        gap = GapCtl().find_gap_by_id(gap_id)
    except GapError:
        return HttpResponseNotFound()

    if not can_see_gap(gap, observer, person):
        return HttpResponseForbidden()

    if gap['full_day']:
        full_day_dates(gap, -1)

    workflow = gap['workflow']

    # подкостыливаем

    gap['to_notify'] = recognize_emails_for_ui(gap['to_notify'])

    form_cls = find_form(workflow)
    form = form_cls(initial=gap)

    result = {}

    person_login = person['login']
    if workflow in TRIP_WORKFLOWS:
        form.make_all_readonly(except_fields=['work_in_absence'])
        forms_person_ids = collect_forms_person_ids([gap])
        form_key = gap.get('form_key')
        if form_key and (observer_id in forms_person_ids.get(form_key, []) or observer.is_superuser):
            result['form_url'] = '/trip/?trip_uuid=%s' % form_key

    result['issues'] = get_unique_issues(gap) if can_see_issues(gap, observer, person_login) else []

    if workflow == 'illness' and gap['has_sicklist']:
        form.make_field_readonly('has_sicklist')

    result.update(form.as_dict())
    result['person'] = dehydrate_person(person)

    fields = list(form_cls.base_fields.keys())
    if workflow not in ['absence', 'learning', 'duty']:
        fields.remove('full_day')

    # убираем галочку "буду работать во время отсутствия"
    # для командировки, конференции, удалённой и офисной работы
    if workflow in TRIP_WORKFLOWS + ['remote_work', 'office_work']:
        fields.remove('work_in_absence')

    # не даем редактировать признак периодичности
    # убрать в STAFF-15493
    if workflow in ['remote_work', 'office_work']:
        fields.remove('is_periodic_gap')

    result['fields'] = {
        gap['workflow']: fields,
    }
    result['subscriptions'] = subscriptions

    return result


def _person_country(person):
    person_country = (
        Staff.objects
        .filter(login=person['login'])
        .values_list('office__city__country__code', flat=True)
        .get()
    )
    return person_country


def _exclude_paid_day_off_workflow_by_office(person_country, workflows):
    if person_country == 'ru':
        return workflows

    return [w for w in workflows if w.workflow != 'paid_day_off']


def _new_gap_get(modifier_id, person, subscriptions):
    person_country = _person_country(person)
    workflows = find_workflows_by_lang(None)
    workflows = _exclude_paid_day_off_workflow_by_office(person_country, workflows)
    workflows = [w for w in workflows if w.workflow != 'duty']

    gap = {}
    w_fields = {}

    for w_cls in workflows:
        workflow = w_cls.workflow
        data = w_cls(modifier_id, person['id'], None).get_base_default(person)
        gap.update(data)
        fields = list(find_form(workflow).base_fields.keys())
        if workflow in TRIP_WORKFLOWS + ['remote_work', 'office_work']:
            fields.remove('work_in_absence')
        if workflow not in ['absence', 'learning']:
            fields.remove('full_day')
        if workflow in ['trip', 'conference_trip']:
            fields.append('trip_details')
        w_fields[workflow] = fields

    w_fields[''] = [
        'workflow',
        'date_from',
        'date_to',
    ]

    form_class = workflow_form_factory([w.workflow for w in workflows])
    form = form_class(initial=gap)
    form.fields_state['workflow'] = sform.REQUIRED

    result = form.as_dict()
    result['fields'] = w_fields
    result['person'] = dehydrate_person(person)
    result['subscriptions'] = subscriptions

    return result
