from django.forms import (
    Form,
    CharField,
    ValidationError,
)

from staff.gap.workflows import WORKFLOWS_REGISTRY


class BaseTemplateForm(Form):
    title = CharField()
    type = CharField()
    workflows = CharField(required=False)
    tag = CharField()
    organization_ids = CharField(required=False)
    city_ids = CharField(required=False)
    langs = CharField(required=False)

    def clean_title(self):
        title = (self.cleaned_data['title'] or '').strip()
        if not title:
            raise ValidationError('Пустое имя — плохое имя, нехорошее.')

        return title

    def clean_type(self):
        type = (self.cleaned_data['type'] or '').strip()
        if not type:
            raise ValidationError('Пустой тип — плохой тип, нехороший.')

        return type

    def clean_tag(self):
        tag = (self.cleaned_data['tag'] or '').strip()
        if not tag:
            raise ValidationError('Пустой тег — плохой тег, нехороший.')

        return tag

    def clean_workflows(self):
        result = []
        workflows = (self.cleaned_data['workflows'] or '').split(',')
        for workflow in workflows:
            workflow = workflow.strip()
            if not workflow:
                continue
            if workflow not in WORKFLOWS_REGISTRY:
                raise ValidationError('Неизвестный тип отсутствия: %s.' % workflow)
            result.append(workflow)

        return result or None

    def clean_organization_ids(self):
        result = []
        organization_ids = (self.cleaned_data['organization_ids'] or '').split(',')
        for organization_id in organization_ids:
            organization_id = organization_id.strip()
            if not organization_id:
                continue
            try:
                organization_id = int(organization_id)
            except Exception:
                raise ValidationError('Непонятный ID организации: %s.' % organization_id)
            result.append(organization_id)

        return result or None

    def clean_city_ids(self):
        result = []
        city_ids = (self.cleaned_data['city_ids'] or '').split(',')
        for city_id in city_ids:
            city_id = city_id.strip()
            if not city_id:
                continue
            try:
                city_id = int(city_id)
            except Exception:
                raise ValidationError('Непонятный ID города: %s.' % city_id)
            result.append(city_id)

        return result or None

    def clean_langs(self):
        result = []
        langs = (self.cleaned_data['langs'] or '').split(',')
        for lang in langs:
            lang = lang.strip()
            if not lang:
                continue
            if lang not in ['ru', 'en']:
                raise ValidationError('Неизвестный язык: %s.' % lang)
            result.append(lang)

        return result or None


class EmailTemplateForm(BaseTemplateForm):
    email_text = CharField(required=False)

    def clean_email_text(self):
        return (self.cleaned_data['email_text'] or '').strip()


class PdfTemplateForm(BaseTemplateForm):
    pdf_head = CharField(required=False)
    pdf_body_start = CharField(required=False)
    pdf_body_end = CharField(required=False)

    def clean_pdf_head(self):
        return (self.cleaned_data['pdf_head'] or '').strip()

    def clean_pdf_body_start(self):
        return (self.cleaned_data['pdf_body_start'] or '').strip()

    def clean_pdf_body_end(self):
        return (self.cleaned_data['pdf_body_end'] or '').strip()


class StIssueTemplateForm(BaseTemplateForm):
    st_issue_queue = CharField()
    st_issue_type = CharField()
    st_issue_components = CharField(required=False)
    st_issue_summary = CharField(required=False)
    st_issue_description = CharField(required=False)

    def clean_st_issue_queue(self):
        return (self.cleaned_data['st_issue_queue'] or '').strip()

    def clean_st_issue_type(self):
        return int(self.cleaned_data['st_issue_type'])

    def clean_st_issue_components(self):
        result = []
        components = (self.cleaned_data['st_issue_components'] or '').split(',')
        for component in components:
            component = component.strip()
            if not component:
                continue
            try:
                component = int(component)
            except Exception:
                raise ValidationError('Непонятный ID компонента: %s.' % component)
            result.append(component)

        return result

    def clean_st_issue_summary(self):
        return (self.cleaned_data['st_issue_summary'] or '').strip()

    def clean_st_issue_description(self):
        return (self.cleaned_data['st_issue_description'] or '').strip()


class StIssueCommentTemplateForm(BaseTemplateForm):
    st_issue_comment_text = CharField(required=False)
    st_issue_comment_state_action_id = CharField(required=False)

    def clean_st_issue_comment_text(self):
        return (self.cleaned_data['st_issue_comment_text'] or '').strip()

    def clean_st_issue_comment_state_action_id(self):
        result = self.cleaned_data['st_issue_comment_state_action_id']
        if result is not None:
            return result.strip() or None
