from datetime import datetime, timedelta

from django.contrib.auth.decorators import login_required, permission_required
from django.shortcuts import render
from django.views.decorators.http import require_http_methods

from staff.lib.utils.date import parse_datetime

from staff.gap.controllers.gap import GapCtl
from staff.gap.workflows.registry import library
from staff.gap.workflows import (
    WORKFLOWS_REGISTRY,
    ALL_WORKFLOWS_ORDER,
)

import logging
logger = logging.getLogger('staff.gap.gap_admin.views.stats_view')


@require_http_methods(['GET'])
@login_required
@permission_required('gap.gap_admin')
def stats(request):
    template_url = 'gap_admin/gaps/stats.html'

    now = datetime.utcnow()

    params = request.GET if request.method == 'GET' else request.POST

    date_from = parse_datetime(params.get('date_from')) or now - timedelta(days=7)
    date_to = parse_datetime(params.get('date_to')) or now
    stats = _get_stats(date_from, date_to)
    workflows = []
    for workflow in ALL_WORKFLOWS_ORDER:
        verbose_name = library.get(WORKFLOWS_REGISTRY[workflow]).verbose_name
        workflows.append({
            'verbose_name': verbose_name,
            'workflow': workflow,
            'stats': stats[workflow] if workflow in stats else _empty_stat()
        })

    context = {
        'workflows': workflows,
        'date_from': date_from.isoformat(),
        'date_to': date_to.isoformat(),
        'total': stats['total'],
    }

    return render(request, template_url, context)


def _get_stats(date_from, date_to):
    # очень линейно написанная функция, но вызываться будет раз в месяц, думаю
    result = {}
    g_ctl = GapCtl()
    fields = {
        'workflow',
        'state',
        'full_day',
        'work_in_absence',
    }
    gaps = g_ctl.find({'$and': [
        {'created_at': {'$gte': date_from}},
        {'created_at': {'$lte': date_to}}
    ]}, fields=fields)

    total = 0
    for gap in gaps:
        total += 1
        data = result.setdefault(gap['workflow'], _empty_stat())
        data['total'] += 1
        if gap['work_in_absence']:
            data['work_in_absence_yes'] += 1
        else:
            data['work_in_absence_no'] += 1
        if gap['full_day']:
            data['full_day_yes'] += 1
        else:
            data['full_day_no'] += 1
        data[gap['state']] += 1

    result['total'] = total

    return result


def _empty_stat():
    return {
        'total': 0,
        'work_in_absence_yes': 0,
        'work_in_absence_no': 0,
        'full_day_yes': 0,
        'full_day_no': 0,
        'new': 0,
        'canceled': 0,
        'confirmed': 0,
        'signed': 0,
    }
