from datetime import date
import logging
from typing import List

from django.conf import settings
from django.core.cache import cache

from staff.lib.calendar import get_holidays

logger = logging.getLogger(__name__)


class OutModeType:
    def __init__(self, calendar_out_mode: str, day_types: List[str]):
        self.calendar_out_mode = calendar_out_mode
        self.day_types = day_types

    def __str__(self):
        day_types = '_'.join(self.day_types)
        return f'{self.calendar_out_mode}_{day_types}'

    def includes_day_type(self, day_type: str) -> bool:
        return not self.day_types or day_type in self.day_types


class OutModes:
    holidays = OutModeType(calendar_out_mode='overrides', day_types=['holiday'])
    weekdays = OutModeType(calendar_out_mode='weekdays', day_types=[])


def get_holidays_by_year(year: int, geo_id: int, out_mode: OutModeType) -> List[date]:
    holidays = cache.get(settings.HOLIDAYS_CACHE_PREFIX.format(geo_id=geo_id, year=year, out_mode=out_mode))
    if holidays is None:
        holidays = get_holidays_from_calendar_api(year, geo_id, out_mode=out_mode)
        cache.set(
            settings.HOLIDAYS_CACHE_PREFIX.format(geo_id=geo_id, year=year, out_mode=out_mode),
            holidays,
            settings.HOLIDAYS_CACHE_TTL,
        )

    return holidays


def get_holidays_from_calendar_api(year: int, geo_id: int, out_mode: OutModeType) -> List[date]:
    holidays_for_geo_ids = get_holidays(
        start_date=date(year, 1, 1),
        end_date=date(year, 12, 31),
        geo_ids=[geo_id],
        out_mode=out_mode.calendar_out_mode,
    )
    holidays = holidays_for_geo_ids[geo_id]

    return [
        holiday['date'] for holiday in holidays
        if out_mode.includes_day_type(holiday['day-type'])
    ]
