import pymongo

import xml.dom.minidom
from datetime import (
    datetime,
    timedelta,
)

from django.conf import settings

from staff.gap.controllers.gap import (
    GapCtl,
    GapQueryBuilder,
)
from staff.gap.workflows.utils import find_workflow
from staff.gap.workflows.choices import LIMITED_STATES


STAFF_DATE_FORMAT = '%d.%m.%Y'
DEFAULT_ENCODING = 'windows-1251'


CURRENT_FIELDS = [
    'id',
    'workflow',
    'gap_type',
    'person_login',
    'person_id',
    'date_from',
    'date_to',
    'comment',
    'full_day',
    'state',
]


def _current_qs(future_lookup, login_list):
    period_from = datetime.utcnow().replace(hour=0, minute=0, second=0)

    gqb = (
        GapQueryBuilder()
        .workflows_states(LIMITED_STATES)
    )

    if future_lookup is None or future_lookup < 0:
        gqb.op_gte('date_to', period_from)
    elif future_lookup == 0:
        gqb.dates_not_strict(period_from, period_from)
    else:
        gqb.dates_not_strict(period_from, period_from + timedelta(days=future_lookup))

    if login_list:
        gqb.person_logins(login_list)

    return GapCtl().find_gaps(query=gqb.query(), fields=CURRENT_FIELDS)


def _to_current_xml(qs, show_description, current_time, encoding):
    if current_time is None:
        current_time = datetime.now()

    encoding = encoding or DEFAULT_ENCODING

    dom = xml.dom.minidom.Document()

    root = dom.createElement('planner')
    dom.appendChild(root)

    absents_container = dom.createElement('absents')
    root.appendChild(absents_container)

    soon_container = dom.createElement('soon')
    root.appendChild(soon_container)

    url_pattern = settings.GAP_URL_PATTERN

    for gap in qs:
        person_login = gap['person_login']

        if gap['full_day']:
            gap['date_to'] = gap['date_to'] - timedelta(days=1)

        absent = dom.createElement('absent')

        workflow = find_workflow(gap['workflow'])

        verbose_type = workflow.verbose_name

        if show_description:
            text_data = dom.createTextNode(gap['comment'])
        else:
            text_data = dom.createTextNode(verbose_type)

        absent.appendChild(text_data)

        attrs = absent.attributes

        attrs['id'] = str(gap['id'])
        attrs['user_id'] = str(gap['person_id'])
        attrs['login'] = person_login
        attrs['mail'] = '%s@yandex-team.ru' % person_login
        attrs['color'] = workflow.color
        attrs['start'] = gap['date_from'].strftime(STAFF_DATE_FORMAT)
        attrs['end'] = gap['date_to'].strftime(STAFF_DATE_FORMAT)
        attrs['url'] = url_pattern % gap['id']
        attrs['subject_id'] = workflow.workflow
        attrs['subject'] = verbose_type

        if gap['date_from'] > current_time:
            soon_container.appendChild(absent)
        else:
            absents_container.appendChild(absent)

    return dom.toxml(encoding)


def get_current_xml(future_lookup=None, login_list=None,
                    show_description=False, current_time=None,
                    encoding=DEFAULT_ENCODING):
    return _to_current_xml(
        _current_qs(future_lookup, login_list),
        show_description,
        current_time,
        encoding,
    )


def _presence_qs(date_from, date_to, login_list):
    gqb = (
        GapQueryBuilder()
        .dates_not_strict(date_from, date_to)
        .workflows_states(LIMITED_STATES)
    )

    if login_list:
        gqb.person_logins(login_list)

    return GapCtl().find_gaps(query=gqb.query(), fields=CURRENT_FIELDS)


def get_presence_dict(date_from, login_list):
    date_to = date_from.replace(hour=0, minute=0, second=0) + timedelta(days=1)

    qs = _presence_qs(date_from, date_to, login_list)

    unavailable = [gap['person_login'] for gap in qs]

    return {
        login: {'is_available': login not in unavailable}
        for login in login_list
    }


def get_gap_list(date_from, date_to, login_list, workflow):
    gqb = (
        GapQueryBuilder()
        .dates_not_strict(date_from, date_to)
        .workflows_states(LIMITED_STATES)
    )

    if login_list:
        gqb.person_logins(login_list)

    if workflow:
        gqb.workflow(workflow)

    fields = [
        'id',
        'workflow',
        'person_login',
        'date_from',
        'date_to',
        'state',
    ]

    sorting = [('date_from', pymongo.ASCENDING)]

    return GapCtl().find_gaps(query=gqb.query(), fields=fields, sorting=sorting)
