from datetime import datetime

from django.http import HttpResponse
from django.views.decorators.http import require_GET

from staff.person.models import Staff

from staff.lib.decorators import responding_json

from staff.gap.old_api.forms import (
    StaffCurrentForm,
    StaffPresenceForm,
    GapListForm,
)
from staff.gap.old_api.controller import (
    get_current_xml,
    get_presence_dict,
    get_gap_list,
)


@require_GET
def current_xml(request):
    form = StaffCurrentForm(request.GET)
    if not form.is_valid():
        return HttpResponse('', status=400)

    ext_str = form.cleaned_data.get('ext', '')

    try:
        ext = bool(int(ext_str))
    except ValueError:
        ext = bool(ext_str.strip())

    return HttpResponse(
        get_current_xml(
            future_lookup=form.cleaned_data.get('future_lookup'),
            login_list=form.cleaned_data.get('login_list'),
            show_description=ext,
            encoding=form.cleaned_data.get('encoding'),
        ),
        content_type='text/xml',
    )


@require_GET
@responding_json
def staff_presence(request):
    form = StaffPresenceForm(data=request.GET)

    if form.is_valid():
        login_list = form.cleaned_data['login_list']
        date_from = form.cleaned_data.get('dt') or datetime.now()
        staff_list = (
            Staff.objects
            .values_list('login', flat=True)
            .filter(login__in=login_list)
        )

        if len(login_list) == staff_list.count():
            return {
                'result': True,
                'staff': get_presence_dict(date_from, login_list),
            }

    return {
        'result': False,
        'error': 'Invalid params',
    }


INVITER_STATES = {
    'new': 'created',
    'confirmed': 'confirmed',
    'signed': 'confirmed',
}


@require_GET
@responding_json
def gap_list(request):
    form = GapListForm(data=request.GET)

    if form.is_valid():
        data = form.cleaned_data
        period_from = datetime.combine(data['period_from'], datetime.min.time())
        period_to = datetime.combine(data['period_to'], datetime.min.time())
        login_list = data['login_list']

        result = {}

        gaps = get_gap_list(period_from, period_to, login_list, data.get('gap_type'))

        for gap in gaps:
            person_login = gap['person_login']
            absences = result.setdefault(person_login, [])
            absences.append({
                'from': gap['date_from'],
                'till': gap['date_to'],
                'state': INVITER_STATES.get(gap['state'], 'new'),
                'type': gap['workflow'],
                'id': gap['id'],
                'staff': person_login,
            })

        return {
            'result': True,
            'gap_list': result,
        }

    return {
        'result': False,
        'error': 'Invalid params',
    }
