from datetime import date, datetime, timedelta, time
import logging
from random import randint
from typing import List, Set, Tuple, Dict

from celery import Task

from django.conf import settings

from staff.celery_app import app
from staff.departments.models import DepartmentStaff, DepartmentRoles
from staff.map.models import City
from staff.lib.db import atomic
from staff.lib.tasks import LockedTask
from staff.lib.utils.list import divide_into_batches
from staff.lib.utils.staff17935 import filter_tigran_recipients
from staff.person.models import Staff, AFFILIATION

from staff.gap.controllers.email_ctl import EmailCtl
from staff.gap.controllers.gap import GapCtl, GapQueryBuilder
from staff.gap.controllers.mail_to_send import EmailToSendCtl, EMAIL_TO_SEND_STATES
from staff.gap.controllers.startrek import StartrekCtl
from staff.gap.controllers.utils import get_chief
from staff.gap.exceptions import MandatoryVacationYearChangedError
from staff.gap.notifications import GapEmailNotification, PeriodicGapEmailNotification
from staff.gap.workflows.choices import GAP_STATES as GS
from staff.gap.workflows import registry
from staff.gap.workflows.duty.sync import sync_from_abc
from staff.gap.workflows.vacation.workflow import VacationWorkflow
from staff.gap.workflows.utils import (
    get_min_mandatory_vacation_date_to,
    get_date_n_business_days_before,
)


logger = logging.getLogger(__name__)


@app.task
class SendMail(Task):
    def run(self, to_send, context, gap_id=None, periodic_gap_id=None):
        assert (gap_id is None) != (periodic_gap_id is None)
        try:
            for to_send_data in to_send:
                template = to_send_data['template']['template']
                if template:
                    template = template.replace('\r', '\n')

                emails = filter_tigran_recipients(context['person_login'], to_send_data['to_send'])
                if not emails:
                    logger.info(
                        'SendMail skipped because all recipients for %s was filtered out',
                        context['person_login'],
                    )
                    continue

                emails.append(settings.GAP_EMAIL_COPY)

                if gap_id is not None:
                    notification = GapEmailNotification(str(gap_id), template, context)

                if periodic_gap_id is not None:
                    notification = PeriodicGapEmailNotification(str(periodic_gap_id), template, context)

                notification.send(**{
                    'recipients': emails,
                    'sender': settings.GAP_EMAIL_SENDER,
                    'headers': gap_mail_headers(
                        context.get('workflow_workflow'),
                        to_send_data['template']['tag'],
                    )
                })
        except Exception as exc:
            if gap_id is not None:
                error_data = f'gap[{gap_id}]'
            if periodic_gap_id is not None:
                error_data = f'periodic_gap[{periodic_gap_id}]'
            logger.exception('Error sending mail of %s', error_data)
            self.retry(countdown=600, exc=exc, max_retries=3)


@app.task(ignore_result=True)
class SendDeferredMail(LockedTask):
    @atomic
    def locked_run(self):
        ets_ctl = EmailToSendCtl()
        emails_to_send = ets_ctl.find_to_send(set_requested=True)
        for email in emails_to_send:
            gap_id = email.get('gap_id')
            periodic_gap_id = email.get('periodic_gap_id')

            assert (gap_id is None) != (periodic_gap_id is None)

            try:
                text = email['text'].replace('\r', '\n')

                if gap_id is not None:
                    notification = GapEmailNotification(str(gap_id), text, {})

                if periodic_gap_id is not None:
                    notification = PeriodicGapEmailNotification(str(periodic_gap_id), text, {})

                notification.send(**{
                    'recipients': email['recipients'] + [settings.GAP_EMAIL_COPY],
                    'sender': settings.GAP_EMAIL_SENDER,
                    'headers': gap_mail_headers(
                        email.get('workflow'),
                        email.get('tag'),
                    ),
                })
                ets_ctl.set_state_one(email['_id'], EMAIL_TO_SEND_STATES.WAS_SENT)
            except Exception:
                if gap_id is not None:
                    error_data = f'gap[{gap_id}]'
                if periodic_gap_id is not None:
                    error_data = f'periodic_gap[{periodic_gap_id}]'
                logger.exception('Error sending mail of %s', error_data)
                try:
                    ets_ctl.set_state_one(email['_id'], EMAIL_TO_SEND_STATES.ERROR)
                except Exception:
                    logger.exception('Error logging error of sending mail of %s', error_data)


@app.task(ignore_result=True)
class CancelDismissedVacationsTask(LockedTask):
    @atomic
    def locked_run(self):
        workflow_cls = registry.library.get('VacationWorkflow')
        vacations = find_vacations_to_cancel()
        for vacation in vacations:
            try:
                workflow_cls.init_to_modify(
                    modifier_id=settings.ROBOT_STAFF_ID,
                    gap_id=vacation['id'],
                ).cancel_gap(
                    send_email=False,
                    issue_comment_tag='cancel_dismissed_vacation',
                )
            except Exception:
                logger.exception('Error canceling vacation[%s]' % vacation['id'])


def gap_mail_headers(workflow, tag):
    headers = {}
    if workflow:
        headers['X-Gap-Workflow'] = workflow
    if tag:
        headers['X-Gap-Tag'] = tag
    return headers


def _person_dissmised_before_vacation(gap, dismiss_dates):
    return gap['date_to'].date() >= dismiss_dates[gap['person_id']]


def find_vacations_to_cancel():
    quit_date_from = datetime.today() - timedelta(days=8)
    quit_date_to = quit_date_from + timedelta(days=7)
    dismissed_persons = Staff.objects.filter(
        is_dismissed=True,
        quit_at__gte=quit_date_from,
        quit_at__lte=quit_date_to,
    ).values('id', 'quit_at')

    dismiss_dates = {p['id']: p['quit_at'] for p in dismissed_persons}

    gqb = (
        GapQueryBuilder()
        .workflow('vacation')
        .person_ids(list(dismiss_dates.keys()))
    )

    fields = [
        'id',
        'person_id',
        'date_to',
        'state',
    ]

    gaps_to_cancel = []

    for gap in GapCtl().find_gaps(query=gqb.query(), fields=fields):
        if gap['state'] == GS.NEW:
            gaps_to_cancel.append(gap)
        elif gap['state'] == GS.CONFIRMED and _person_dissmised_before_vacation(gap, dismiss_dates):
            gaps_to_cancel.append(gap)

    return gaps_to_cancel


@app.task(ignore_result=True)
class RemindFinishedTripsTask(LockedTask):
    @atomic
    def locked_run(self):
        remind_to = datetime.utcnow()
        remind_from = remind_to - timedelta(days=3)
        workflows_cls = [
            registry.library.get('TripWorkflow'),
            registry.library.get('ConferenceTripWorkflow'),
        ]
        for workflow_cls in workflows_cls:
            gqb = (
                GapQueryBuilder()
                .workflow(workflow_cls.workflow)
                .state(GS.CONFIRMED)
                .date_in_range('date_to', remind_from, remind_to)
                .sent_aftertrip_reminder(False)
            )

            gaps = list(GapCtl().find_gaps(query=gqb.query(), fields=['id', 'person_login']))
            login_list = [gap['person_login'] for gap in gaps]

            country_by_login = dict(
                Staff.objects
                .filter(login__in=login_list)
                .values_list('login', 'organization__country_code')
            )

            for gap in gaps:
                # если организация не из России, то подготовить письмо для отправки
                if country_by_login[gap['person_login']] != settings.RUSSIA_CODE:
                    workflow = workflow_cls.init_to_modify(
                        modifier_id=settings.ROBOT_STAFF_ID,
                        gap_id=gap['id'],
                    )
                    workflow.deffer_finish_reminder()


@app.task(ignore_result=True)
class VacationStatementReminderTask(LockedTask):
    @atomic
    def locked_run(self):
        date_from = datetime.combine(datetime.utcnow().date(), time.min) + timedelta(days=15)
        date_to = date_from + timedelta(days=7)
        vacation_wf = registry.library.get('VacationWorkflow')
        gqb = (
            GapQueryBuilder()
            .workflow(vacation_wf.workflow)
            .date_in_range('date_from', date_from, date_to)
            .commented_vacation_statement_reminder(False)
            .states([GS.NEW, GS.CONFIRMED])
        )

        for gap in GapCtl().find_gaps(query=gqb.query(), fields=['id', 'master_issue']):
            try:
                ctl = StartrekCtl(issue_key=gap['master_issue'],)
                if ctl.issue.status.key in ['new', 'approved']:
                    wf = vacation_wf.init_to_modify(
                        modifier_id=settings.ROBOT_STAFF_ID,
                        gap_id=gap['id'],
                    )
                    wf.issue_comment('vacation_statement_reminder', login_to_call=wf.gap['person_login'])
                    wf.gap['commented_vacation_statement_reminder'] = True
                    wf._update()
            except Exception:
                logger.error('Error trying to remind vacation statement: gap[%s]' % gap['id'])


@app.task(ignore_result=True)
class VacationTicketCreateRetryTask(LockedTask):
    @atomic
    def locked_run(self):
        date_to = datetime.combine(datetime.utcnow().date(), time.min) - timedelta(minutes=10)
        date_from = date_to - timedelta(days=60)

        vacation_workflow = registry.library.get('VacationWorkflow')
        gqb = (
            GapQueryBuilder()
            .workflow(vacation_workflow.workflow)
            .date_in_range('created_at', date_from, date_to)
            .append({"master_issue": {"$exists": False}})
            .states([GS.NEW, GS.CONFIRMED])
        )
        vacation_gaps_wo_issues = GapCtl().find_gaps(query=gqb.query(), fields=['id', 'is_selfpaid', 'state'])
        result = {'created': 0, 'failed': 0}

        for gap in vacation_gaps_wo_issues:
            vacation = vacation_workflow.init_to_modify(
                modifier_id=settings.ROBOT_STAFF_ID,
                gap_id=gap['id'],
            )
            tag = 'new_selfpaid_vacation' if gap.get('is_selfpaid', False) else 'new_vacation'
            try:
                vacation.new_issue(tag)
                if vacation.gap['state'] == 'confirmed':
                    vacation.issue_comment('confirm_vacation', forbidden_statuses=['closed', 'cancelled'])
                result['created'] += 1
            except Exception:
                logger.error('Retry failed to create vacation ticket for gap %s', gap['id'])
                result['failed'] += 1

        logger.info(
            'VacationTicketCreateRetryTask result: created: %s, failed %s',
            result['created'],
            result['failed'],
        )


@app.task(ignore_result=True)
class VacationApprovalReminderTask(LockedTask):
    @atomic
    def locked_run(self):
        vacation_wf = registry.library.get('VacationWorkflow')

        days_when_remind = [10, 15, 38]
        for day in days_when_remind:
            date_from = datetime.combine(datetime.utcnow().date(), time.min) + timedelta(days=day)
            date_to = datetime.combine(date_from.date(), time.max)
            gqb = (
                GapQueryBuilder()
                .workflow(vacation_wf.workflow)
                .date_in_range('date_from', date_from, date_to)
                .last_day_when_commented_vacation_approval_reminder(day)
                .states([GS.NEW])
            )
            for gap in GapCtl().find_gaps(query=gqb.query(), fields=['id', 'master_issue']):
                try:
                    ctl = StartrekCtl(issue_key=gap['master_issue'],)
                    if ctl.issue.status.key in ['new']:
                        wf = vacation_wf.init_to_modify(
                            modifier_id=settings.ROBOT_STAFF_ID,
                            gap_id=gap['id'],
                        )
                        chief = get_chief(wf.gap['person_login'], chief_fields=['login'])
                        wf.issue_comment('vacation_approval_reminder', login_to_call=chief.get('login'))
                        wf.gap['last_day_when_commented_vacation_approval_reminder'] = day
                        wf._update()
                except Exception:
                    logger.exception('Error trying to remind vacation approval: gap[%s]' % gap['id'])


@app.task(ignore_result=True)
class SyncDutyGapsTask(LockedTask):
    def locked_run(self, service_id: int = None):
        """
        Синхронизация с abc по конкретному сервису
        или в случае если service_id=None то полностью по всем сервисам
        """
        try:
            sync_from_abc(service_id)
        except Exception as e:
            logger.exception('Error sync gap from abc. Reason: "%s"', str(e))

    def get_lock_name(self, service_id: int = None) -> str:
        task_name = self.__class__.__name__
        service_id = service_id or 'ALL_SERVICES'
        return f'{task_name}__{service_id}'


@app.task
def retry_all_tasks():
    from staff.gap.controllers.gap_tasks import GapTasks
    GapTasks.retry_all()


@app.task
class CreateMandatoryVacations(LockedTask):
    def locked_run(self):
        try:
            staff_with_geo_ids = get_staff_with_geo_ids_for_mandatory_vacations()
        except Exception:
            logger.exception('Failed to start a task CreateMandatoryVacations: can not get staff with geo_ids')
            return

        batch_size = settings.MANDATORY_VACATION_BATCH_SIZE

        for batch in divide_into_batches(staff_with_geo_ids, batch_size):
            CreateMandatoryVacationsForIds.delay(batch)


def get_staff_with_geo_ids_for_mandatory_vacations() -> List[tuple]:
    logins_with_mandatory_vacation = get_logins_with_mandatory_vacation()
    staff_with_geo_ids = (
        Staff.objects
        .filter(
            is_dismissed=False,
            is_robot=False,
            organization__country_code__in=settings.MANDATORY_VACATION_COUNTRIES,
            vacation__gte=settings.MANDATORY_VACATION_REQUIRED_VACATION,
            affiliation=AFFILIATION.YANDEX,
        )
        .exclude(
            login__in=logins_with_mandatory_vacation,
        )
        .values_list('id', 'office__city__geo_id')
    )
    return list(staff_with_geo_ids)


@app.task
class CreateMandatoryVacationsForIds(Task):
    def run(self, staff_with_geo_ids: list):
        for person_id, person_geo_id in staff_with_geo_ids:
            workflow = VacationWorkflow(modifier_id=settings.ROBOT_STAFF_ID, person_id=person_id)

            next_year = datetime.now().year + 1
            min_start_date = date(year=next_year, **settings.MANDATORY_VACATION_MIN_START_DATE)
            max_start_date = date(year=next_year, **settings.MANDATORY_VACATION_MAX_START_DATE)
            vacation_date_from, vacation_date_to = generate_mandatory_vacation_dates(
                min_start_date,
                max_start_date,
                person_geo_id,
            )
            deadline = datetime.combine(
                get_date_n_business_days_before(
                    vacation_date_from,
                    settings.MANDATORY_VACATION_DEADLINE_DAYS,
                    person_geo_id,
                ),
                time.min,
            )
            try:
                with atomic():
                    workflow.new_gap(
                        {
                            'mandatory': True,
                            'full_day': True,
                            'date_from': datetime.combine(vacation_date_from, time.min),
                            'date_to': datetime.combine(vacation_date_to, time.min),
                            'work_in_absence': False,
                            'comment': '',
                            'to_notify': [],
                            'is_selfpaid': False,
                            'countries_to_visit': [],
                            'deadline': deadline,
                            'geo_id': person_geo_id,
                        },
                    )
            except Exception:
                logger.exception('Error creating mandatory vacation for person_id = %s', person_id)


def generate_mandatory_vacation_dates(
    min_start_date: date,
    max_start_date: date,
    geo_id: int,
) -> Tuple[date, date]:

    days_range = (max_start_date - min_start_date).days
    vacation_start = min_start_date + timedelta(days=randint(0, days_range))
    vacation_finish = get_min_mandatory_vacation_date_to(vacation_start, geo_id)

    return vacation_start, vacation_finish


def get_logins_with_mandatory_vacation() -> Set[str]:
    next_year = datetime.now().year + 1
    vacations = GapCtl().find_gaps(
        {
            'workflow': 'vacation',
            'mandatory': True,
            'date_from': {
                '$gte': datetime(next_year, 1, 1),
            },
            'date_to': {
                '$lte': datetime(next_year + 1, 1, 1),
            },
        },
        fields=['person_login'],
    )
    maternity_vacations = GapCtl().find_gaps(
        {
            'workflow': 'maternity',
            'date_to': {
                '$gte': datetime(next_year, 12, 1),
            },
            'date_from': {
                '$lte': datetime(next_year, 5, 1),
            },
        },
        fields=['person_login'],
    )

    return set([v['person_login'] for v in vacations]) | set([v['person_login'] for v in maternity_vacations])


@app.task
class MoveOrApproveMandatoryVacations(LockedTask):
    def locked_run(self):

        geo_ids = City.objects.order_by().values_list('geo_id', flat=True).distinct()
        for geo_id in geo_ids:
            MoveOrApproveMandatoryVacationsForGeoIds.delay(geo_id)


@app.task
class MoveOrApproveMandatoryVacationsForGeoIds(Task):
    def run(self, geo_id):
        now = datetime.now()

        vacations = GapCtl().find_gaps(
            {
                'geo_id': geo_id,
                'deadline': {
                    '$lte': now,
                },
                'mandatory': True,
                'state': GS.NEW,
            },
        )

        if vacations.count() == 0:
            return

        new_date_from, new_date_to, new_deadline = get_data_to_move_gap(deadline=now.date(), geo_id=geo_id)

        for vacation in vacations:
            try:
                with atomic():
                    gap = VacationWorkflow.init_to_modify(settings.ROBOT_STAFF_ID, gap=vacation)
                    gap.edit_gap(
                        {
                            'date_from': new_date_from,
                            'date_to': new_date_to,
                            'deadline': new_deadline,
                        },
                    )
                    gap._move_mandatory_vacation_email()
            except MandatoryVacationYearChangedError:
                # если произошла такая ошибка, то при переносе отпуска меняется год
                # => его нужно автоматически согласовать
                VacationWorkflow.init_to_modify(settings.ROBOT_STAFF_ID, gap=vacation).confirm_gap()
            except Exception:
                logger.exception('Error moving vacation for %s', vacation.get('person_login', ''))


def get_data_to_move_gap(deadline: date, geo_id: int) -> Tuple[datetime, datetime, datetime]:
    new_date_from = datetime.combine(
        deadline + timedelta(settings.MANDATORY_VACATION_OFFSET + settings.MANDATORY_VACATION_DEADLINE_DAYS),
        time.min,
    )
    new_date_to = datetime.combine(
        get_min_mandatory_vacation_date_to(new_date_from.date(), geo_id=geo_id),
        time.min,
    )
    new_deadline = datetime.combine(
        get_date_n_business_days_before(
            new_date_from,
            settings.MANDATORY_VACATION_DEADLINE_DAYS,
            geo_id,
        ),
        time.min,
    )
    return new_date_from, new_date_to, new_deadline


@app.task
def remind_about_upcoming_mandatory_vacations():
    reminders = settings.MANDATORY_VACATION_REMINDERS
    for reminder in reminders:
        vacations_to_remind = get_vacations_to_remind(reminder, mandatory_only=True)
        logins_to_staff_map = get_logins_to_staff_map(
            logins=[v['person_login'] for v in vacations_to_remind],
        )
        department_ids_to_chiefs_map = get_department_ids_to_chiefs_map(
            deps=[st['department_id'] for st in logins_to_staff_map.values()],
        )

        for vacation in vacations_to_remind:
            person_login = vacation['person_login']
            person_dict = logins_to_staff_map[person_login]

            try:
                send_reminder_mail_about_upcoming_vacation(
                    vacation=vacation,
                    person_dict=person_dict,
                    chief_dict=department_ids_to_chiefs_map[person_dict['department_id']],
                    tag=f'remind_about_mandatory_vacation_{reminder}_days_before',
                )
            except Exception:
                logger.exception(
                    'Error sending mandatory vacation reminder to %s, gap_id=%s',
                    person_dict.get('login'),
                    vacation.get('id'),
                )
            else:
                update_vacation_after_reminder_sent(vacation, reminder)


@app.task
def remind_about_upcoming_vacations():
    reminder = settings.VACATION_REMINDER
    vacations_to_remind = get_vacations_to_remind(reminder)
    logins_to_staff_map = get_logins_to_staff_map(
        logins=[v['person_login'] for v in vacations_to_remind],
    )
    department_ids_to_chiefs_map = get_department_ids_to_chiefs_map(
        deps=[st['department_id'] for st in logins_to_staff_map.values()],
    )

    for vacation in vacations_to_remind:
        person_login = vacation['person_login']
        person_dict = logins_to_staff_map[person_login]
        try:
            send_reminder_mail_about_upcoming_vacation(
                vacation=vacation,
                person_dict=person_dict,
                chief_dict=department_ids_to_chiefs_map[person_dict['department_id']],
                tag='remind_about_vacation_3_days_before',
            )
        except Exception:
            logger.exception(
                'Error sending vacation reminder to %s, gap_id=%s',
                person_dict.get('login'),
                vacation.get('id'),
            )
        else:
            update_vacation_after_reminder_sent(vacation, reminder)


def get_vacations_to_remind(reminder: int, mandatory_only: bool = False) -> List[dict]:
    gap_date = datetime.combine(
        date.today() + timedelta(reminder),
        time.min,
    )
    query = {
        'workflow': 'vacation',
        'date_from': {
            '$gte': gap_date,
            '$lte': gap_date + timedelta(1),
        },
    }
    if mandatory_only:
        query['mandatory'] = True

    upcoming_vacations = list(GapCtl().find_gaps(query))

    return [
        vacation for vacation in upcoming_vacations
        if reminder not in vacation.get('reminders_sent', [])
    ]


def get_logins_to_staff_map(logins: List[str]) -> Dict[str, dict]:
    staff = (
        Staff.objects
        .filter(login__in=logins)
        .values(
            'id',
            'first_name',
            'last_name',
            'first_name_en',
            'last_name_en',
            'login',
            'work_email',
            'department_id',
        )
    )
    return {
        st['login']: st
        for st in staff
    }


def get_department_ids_to_chiefs_map(deps: List[int]) -> Dict[int, dict]:
    chiefs = (
        DepartmentStaff.objects.filter(
            department_id__in=set(deps),
            role_id=DepartmentRoles.CHIEF.value,
        )
        .values('department_id', 'staff_id', 'staff__login', 'staff__work_email')
    )
    return {
        ch['department_id']: {
            'login': ch['staff__login'],
            'id': ch['staff_id'],
            'work_email': ch['staff__work_email'],
        }
        for ch in chiefs
    }


def send_reminder_mail_about_upcoming_vacation(vacation: dict, person_dict: dict, chief_dict: dict, tag: str):
    """
    У person_dict нужны ключи id, first_name, last_name, first_name_en, last_name_en, login, work_email
    У chief_dict – id, login, work_email
    """

    EmailCtl(
        data=vacation,
        approver=chief_dict,
    ).notify(
        person=person_dict,
        modifier=person_dict,
        workflow='vacation',
        tag=tag,
        to_notify_overwrite=[person_dict['work_email']],
    )


def update_vacation_after_reminder_sent(vacation: dict, reminder: int):
    vacation_reminders_sent = vacation.get('reminders_sent', [])
    vacation_reminders_sent.append(reminder)
    vacation['reminders_sent'] = vacation_reminders_sent
    GapCtl().update_gap(settings.ROBOT_STAFF_ID, vacation)


@app.task
class ImportMandatoryVacations(Task):

    def run(self, vacations_data: Dict[str, dict]):
        failed_logins = []

        for login, vacation_data in vacations_data.items():
            date_from, date_to, deadline, geo_id = self._get_vacation_params(vacation_data)
            try:
                self.create_mandatory_vacation(vacation_data['id'], date_from, date_to, deadline, geo_id)
            except Exception:
                logger.exception('Error creating mandatory vacation for %s', login)
                failed_logins.append(login)

        if failed_logins:
            logger.error('Failed to create mandatory vacations for logins %s', failed_logins)

    @staticmethod
    def _get_vacation_params(vacation_params: dict) -> Tuple[datetime, datetime, datetime, int]:
        geo_id = vacation_params['geo_id']
        date_from = datetime.strptime(
            vacation_params['date_from'],
            settings.MANDATORY_VACATIONS_IMPORT_FILE_DATE_FORMAT,
        )
        date_to = (
            datetime.strptime(
                vacation_params['date_to'],
                settings.MANDATORY_VACATIONS_IMPORT_FILE_DATE_FORMAT,
            )
            if vacation_params.get('date_to')
            else datetime.combine(
                get_min_mandatory_vacation_date_to(date_from.date(), geo_id),
                time.min,
            )
        )
        deadline = datetime.combine(
            get_date_n_business_days_before(
                date_from.date(),
                settings.MANDATORY_VACATION_DEADLINE_DAYS,
                geo_id,
            ),
            time.min,
        )

        return date_from, date_to, deadline, geo_id

    @staticmethod
    def create_mandatory_vacation(
        person_id: int,
        date_from: datetime,
        date_to: datetime,
        deadline: datetime,
        geo_id: int
    ):
        workflow = VacationWorkflow.init_to_new(
            modifier_id=settings.ROBOT_STAFF_ID,
            person_id=person_id,
        )

        with atomic():
            workflow.new_gap(
                {
                    'mandatory': True,
                    'full_day': True,
                    'date_from': date_from,
                    'date_to': date_to,
                    'work_in_absence': False,
                    'comment': '',
                    'to_notify': [],
                    'is_selfpaid': False,
                    'countries_to_visit': [],
                    'deadline': deadline,
                    'geo_id': geo_id,
                },
            )
            workflow.confirm_gap()
